/*	HTTP_ImageInputStream

PIRL CVS ID: HTTP_ImageInputStream.java,v 1.14 2012/04/16 06:10:20 castalia Exp

Copyright (C) 2006-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Image_Tools;

import PIRL.Strings.Words;
import PIRL.Strings.String_Buffer;

import javax.imageio.stream.ImageInputStreamImpl;
import java.net.URL;
import java.net.MalformedURLException;
import java.net.UnknownHostException;
import java.net.ProtocolException;
import java.net.URLEncoder;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.io.BufferedWriter;
import java.io.OutputStreamWriter;
import java.io.BufferedReader;
import java.io.InputStream;
import java.io.IOException;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.BufferOverflowException;
import java.lang.NumberFormatException;
import java.lang.IllegalArgumentException;
import java.util.Vector;

/**	An <i>HTTP_ImageInputStream</i> provides an <i>ImageInputStream</i>
	interface to a remote source file accessed by a URL reference.
<p>
	The URL used to reference the source file must use the http protocol.
	Any HTTP server compatible with version 1.1 of the protocol should
	be capable of providing selective access to the source file.
<p>
	The source file data is buffered using a sliding window that is
	mapped to an arbitrary location offset of the source file. Data is
	fetched on demand using a Content_Range request to the server from
	which the response data content is placed at the appropriate position
	in the buffer. During normal read operations when the end of buffered
	data is reached a relatively small input increment is fetched into
	the buffer. When the amount of data has reached the buffer capacity
	the buffer "slides" forward, dropping data from the beginning of the
	buffer Arbitrary random access seeks within the source file are
	supported. If the seek location is currently buffered, no data fetch
	is needed, of course. Seeks beyond the range of buffered data to a
	"nearby" location will retain as much buffered data as possible to
	optimize short seeks.
<p>
	If the connection to the server is lost repeated attempts are made
	to reconnect. Reconnecting to the server does cause any loss of
	buffered data.
<p>
	@author		Bradford Castalia UA/PIRL
	@version	1.14
	@see		javax.imageio.stream.ImageInputStream
*/
public class HTTP_ImageInputStream
	extends ImageInputStreamImpl
{
public static final String
	ID = "PIRL.Image_Tools.HTTP_ImageInputStream (1.14 2012/04/16 06:10:20)";

//------------------------------------------------------------------------------
//	Data source

/**	The HTTP version identifier used in communication with the server.
*/
public static final String
	HTTP_VERSION			= "HTTP/1.1";

/**	The character encoding for messages sent to the server.
*/
public static final String
	ENCODING_SCHEME			= "UTF-8";

/**	The default {@link #User_Agent(String) User Agent} reported to the server.
*/
public static final String
	DEFAULT_USER_AGENT		= "HTTP_ImageInputStream 1.14 2012/04/16 06:10:20";

/**	The default port to use for communication with the server.
<p>
	The URL used to construct the HTTP_ImageInputStream may specify
	a different port to be used.
*/
public static final int
	DEFAULT_PORT			= 80;

/**	The amount of time (ms) to wait for a server connection.
*/
public static final int
	DEFAULT_CONNECT_TIMEOUT	= 5000;

/**	The amount of time (ms) to wait before trying to reconnect with the server.
*/
public static final long
	RECONNECT_WAIT			= 3000;

/**	The maximum number of repeated server reconnect tries.
*/
public static final int
	MAX_RECONNECT_ATTEMPTS	= 5;

/**	HTTP {@link #Status() Status} code when the source can not be found
	on the server.
*/
public static final int
	SOURCE_NOT_FOUND_STATUS	= 404;

/**	HTTP {@link #Status() Status} code value below which the request
	is deemed successful.
*/
public static final int
	SUCCESS_STATUS_LIMIT	= 300;

private URL
	Source_URL				= null;

private static String
	Default_User_Agent		= DEFAULT_USER_AGENT;
private String
	User_Agent				= Default_User_Agent,
	Server_Hostname			= null,
	Source_Pathname			= null;

private InetSocketAddress
	Server_Address			= null;
private Socket
	Server_Socket			= null;
	
private BufferedWriter
	Server_Writer			= null;
private static final int
	HEADERS_BUFFER_SIZE		= 512;

private InputStream
	Server_Input			= null;

/**	Source size value if no source has been found.
*/
protected long
	NO_SOURCE_SIZE			= Long.MAX_VALUE;

/**	Total size (bytes) of the source data file.
<p>
	The value is initialized to {@link #NO_SOURCE_SIZE} to indicate that
	the size is unknown. When HTTP response {@link #Headers() headers}
	are received an attempt is made to get the {@link #Source_Size()
	size of the source file} if it is not already known.
*/
protected long
	Source_Size				= NO_SOURCE_SIZE;	// Get the size from the server.

/*	!!! Workaround !!!

	The reset mehod of ImageInputStreamImpl throws an exception if the
	position popped from the markByteStack is located before the
	flushedPos (beginning of the buffer). This implementation allows
	seeking to a location before the beginning of the buffer. The
	workaround is to leave the flushedPos at zero and use an alternate
	variable for the buffer stream location.
*/
private long
	Buffer_Location			= 0;


//!	Total number of bytes transmitted to/from the sever.
private long
	Total_Sent				= 0,
	Total_Received			= 0;

//------------------------------------------------------------------------------
//	HTTP headers

private Vector
	HTTP_Headers			= new Vector ();

private int
	HTTP_Status				= 0;

private boolean
	Logging					= false;

/**	When {@link #Logging(boolean) Logging} is enabled, the prefix to
	print before each request message line that is reported sent to the
	server.
*/
public static final String
	LOG_OUTPUT_PREFIX		= "<< ";

/**	When {@link #Logging(boolean) Logging} is enabled, the prefix to
	print before each response header line that is reported received
	from the server.
*/
public static final String
	LOG_INPUT_PREFIX		= "    >> ";


private static final String
	WHITESPACE					= " \t",
	HTTP_VERSION_DELIMITERS		= "/" + WHITESPACE,
	FIELD_CONTENT_DELIMITERS	= ":;,&" + WHITESPACE,
	VALUE_DELIMITERS			= "=" + WHITESPACE,
	EOL							= "\r\n";

//------------------------------------------------------------------------------
//  DEBUG control
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTORS	= 1 << 0,
	DEBUG_ACCESSORS		= 1 << 1,
	DEBUG_SEND			= 1 << 2,
	DEBUG_HELPERS		= 1 << 3,
	DEBUG_HEADERS		= 1 << 4,
	DEBUG_DATA			= 1 << 5,
	DEBUG_RECEIVE		= 1 << 6,
	DEBUG_CONNECT		= 1 << 7,
	DEBUG_BUFFERS		= 1 << 8,
	DEBUG_IIS			= 1 << 9,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;
//	DEBUG		    	= DEBUG_ALL & ~DEBUG_DATA & ~DEBUG_HEADERS;

/*==============================================================================
	Constructors
*/
/**	Constructs an HTTP_ImageInputStream on a URL.
<p>
	After a successful {@link #Connect(URL) connect}ion to the server has
	been established, a data storage buffer is {@link #Allocate_Buffer(int)
	allocated} with the {@link #DEFAULT_BUFFER_SIZE} capacity. Then the
	validity of the source is {@link #Check_Source() checked}. If the
	source can not be confirmed the connection to the server is closed.
	Otherwise, if the server response to the source check contains a
	{@link #Content_Length() content length} that is used to set the
	size of the source file.
<p>
	@param	url	The URL to use to access the source file.
	@throws	IllegalArgumentException If the URL does not specify the
		http protocol nor provide a source file pathname.
	@throws	UnknownHostException If the hostname of the URL can not
		be resolved to an internet address.
	@throws	FileNotFoundException	If the source was not found on the server.
	@throws	ProtocolException	If an HTTP {@link #Status() Status}
		protocol error was returned by the server.
	@throws	IOException If the server connection could not be established.
*/
public HTTP_ImageInputStream
	(
	URL		url
	)
	throws IOException, UnknownHostException, IllegalArgumentException
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	{
	System.out.println
		(">>> HTTP_ImageInputStream: URL " + url);
	Logging (true);
	}
Connect (url);
Allocate_Buffer (DEFAULT_BUFFER_SIZE);

int
	source_status = Check_Source ();
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("    Source status: " + source_status);
int
	attempt = 0;
while (source_status >= SUCCESS_STATUS_LIMIT &&
		attempt < MAX_RECONNECT_ATTEMPTS)
	{
	if (Redirect ())
		{
		if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
			System.out.println
				("    Redirected to " + Source_URL);
		source_status = Check_Source ();
		if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
			System.out.println
				("    Source status: " + source_status);
		++attempt;
		}
	else
		break;
	}
if (source_status >= SUCCESS_STATUS_LIMIT)
	{
	close ();
	if (source_status == SOURCE_NOT_FOUND_STATUS)
		throw new FileNotFoundException (ID + '\n'
			+ "URL not found: " + url);
	else if (attempt == MAX_RECONNECT_ATTEMPTS)
		throw new ProtocolException (ID + '\n'
			+ "Too many server redirects for source at " + url + '\n'
			+ "Last attempt was to URL " + Source_URL);
	else
		throw new ProtocolException (ID + '\n'
			+ "Status " + source_status + " for source at " + url);
	}
long
	length = Content_Length ();
if (length >= 0)
	Source_Size = length;

if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream: Source size = " + Source_Size);
}

/*==============================================================================
	Accessors
*/
/**	Get the URL used to access the source.
<p>
	@return	The source URL.
*/
public URL Source_URL ()
{return Source_URL;}

/**	Get the source server hostname.
<p>
	@return	The hostname String for the server.
*/
public String Server_Hostname ()
{return Server_Hostname;}

/**	Get the source pathname.
<p>
	@return	The pathname String where the source to be found by the server.
*/
public String Source_Pathname ()
{return Source_Pathname;}

/**	Get the HTTP user agent name reported to the server.
<p>
	@return	The user agent name String.
	@see	#User_Agent(String)
*/
public String User_Agent ()
{return User_Agent;}

/**	Set the HTTP user agent name to be reported to the server.
<p>
	Every time a request is sent to the server a "User-Agent" message
	is included. This information can be used by the server to gather
	statistics on it's use by the identified client agent. When an
	HTTP_ImageInputStream object is constructed the user agent is
	set as the {@link #Default_User_Agent}.
<p>
	@param	name	The user agent name String. If this is null, the
		{@link #Default_User_Agent} will be assigned.
	@return	This HTTP_ImageInputStream.
*/
public HTTP_ImageInputStream User_Agent
	(
	String	name
	)
{
if (name == null ||
	name.length () == 0)
	User_Agent = Default_User_Agent;
else
	User_Agent = name;
return this;
}

/**	Get the default user agent that will be used with new
	HTTP_ImageInputStream objects.
<p>
	@return	The default user agent String.
	@see	#User_Agent(String)
*/
public static String Default_User_Agent ()
{return Default_User_Agent;}

/**	Set the default user agent that will be used with new
	HTTP_ImageInputStream objects.
<p>
	@param	name	The user agent name String. If this is null, the
		{@link #DEFAULT_USER_AGENT} will be assigned.
	@see	#User_Agent(String)
*/
public static void Default_User_Agent
	(
	String	name
	)
{
if (name == null ||
	name.length () == 0)
	Default_User_Agent = DEFAULT_USER_AGENT;
else
	Default_User_Agent = name;
}

/**	Get the total number of bytes sent to the server so far.
<p>
	@return	A long value that is the number of bytes that have been
		sent to the server.
*/
public long Total_Sent ()
{return Total_Sent;}

/**	Get the total number of bytes received from the server so far.
<p>
	@return	A long value that is the number of bytes that have been
		received from the server.
*/
public long Total_Received ()
{return Total_Received;}

/**	Check if server communication logging is enabled.
<p>
	@return	true if logging is enabled; false otherwise.
	@see	#Logging(boolean)
*/
public boolean Logging ()
{return Logging;}

/**	Enable or disable server communication logging.
<p>
	When logging is enabled, each message {@link #Send(String) sent} to
	the server is printed to stdout preceded by the {@link
	#LOG_OUTPUT_PREFIX} String, and each {@link #Headers() header}
	response line is printed preceded by the {@link #LOG_INPUT_PREFIX}
	String. In addition, the amount of data content {@link #Receive()
	receive}d is printed on a ling after a LOG_INPUT_PREFIX.
<p>
	@param	enabled	true if logging is to be enabled; false otherwise.
	@return	This HTTP_ImageInputStream.
*/
public HTTP_ImageInputStream Logging
	(
	boolean	enabled
	)
{
Logging = enabled;
return this;
}

/*==============================================================================
	ImageInputStream
*/
//	Required method.
/**	Read a source byte.
<p>
	The {@link #getBitOffset() bit offset} is set to zero.
<p>
	@return	The next byte value from the source. This will be -1
		if the current source location is end of file.
	@throws	IOException If access to the source is closed or there was a
		problem obtaining source data.
*/
public int read ()
	throws IOException
{
if ((DEBUG & DEBUG_IIS) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.read:\n"
		+"    stream location " + Stream_Location () + " (" + streamPos + ')');
int
	datum = -1;
if (Stream_Location () < Source_Size)
	{
	if (Image_Buffer.remaining () > 0 ||
		Fetch (INPUT_INCREMENT) > 0)
		{
		//	Required ImageInputStream functionality.
		bitOffset = 0;
		++streamPos;
		datum = (int)Image_Buffer.get () & 0xFF;
		}
	}
if ((DEBUG & DEBUG_IIS) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.read: 0x" + Integer.toHexString (datum));
return datum;
}

//	Required method.
/**	Read a sequence of source bytes and store them in an array.
<p>
	The number of bytes actually read may be less than the specified
	length. The length will be reduced to the smaller of the amount
	of space in the data array following the offset or the amount of
	buffered data remaining after an attempt to provide length bytes.
<p>
	The {@link #getBitOffset() bit offset} is set to zero.
<p>
	The current source input location is advanced by the amount of
	data read.
<p>
	@param	data	The byte array into which the data is to be stored.
	@param	offset	The offset of the array where the first byte read is
		to be stored.
	@param	length	The number of bytes to read.
	@return	The number of bytes read and stored. This will be -1 if the
		current source location is end of file.
	@throws	NullPointerException If the data array is null.
	@throws IndexOutOfBoundsException If the offset or length are
		negative, or the offset is greater than the length of the data
		array.
	@throws	IOException If access to the source is closed or there was a
		problem obtaining source data.
	@see	#Fetch(int)
*/
public int read
	(
	byte[]	data,
	int		offset,
	int		length
	)
	throws IOException
{
if ((DEBUG & DEBUG_IIS) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.read: data"
			+ data + ", size " + data.length
			+ ", offset " + offset + ", length " + length + '\n'
		+"    stream location " + Stream_Location () + " (" + streamPos + ')');
int
	amount = -1;
if (Stream_Location () < Source_Size)
	{
	if (data == null)
		throw new NullPointerException (ID + '\n'
			+ "Attempt to read into a null data array.");
	if (length < 0 ||
		offset < 0 ||
		offset >= data.length)
		throw new IndexOutOfBoundsException (ID + '\n'
			+ "Attempt to read into array of length " + data.length
				+ " at offset " + offset
				+ " for amount " + length + '.');
	if ((offset + length) > data.length)
		{
		length = data.length - offset;
		if ((DEBUG & DEBUG_IIS) != 0)
			System.out.println
				("    length clipped to array amount after offset: " + length);
		}
	int
		additional = length - Image_Buffer.remaining ();
	if (additional >= 0)
		{
		if ((DEBUG & DEBUG_IIS) != 0)
			System.out.println
				("    additional buffer amount wanted: " + additional);
		if (Data_End_Location () < Source_Size)
			{
			//	Round up to the nearest INPUT_INCREMENT.
			additional += INPUT_INCREMENT - (additional % INPUT_INCREMENT);
			if (Fetch (additional) <= 0)
				{
				if ((DEBUG & DEBUG_IIS) != 0)
					System.out.println
						("<<< HTTP_ImageInputStream.read: " + amount);
				return amount;
				}
			}
		}
	if (length > Image_Buffer.remaining ())
		{
		length = Image_Buffer.remaining ();
		if ((DEBUG & DEBUG_IIS) != 0)
			System.out.println
				("    length clipped to buffer amount remaining: " + length);
		}

	//	Required ImageInputStream functionality.
	bitOffset = 0;
	streamPos += length;
	Image_Buffer.get (data, offset, length);
	amount = length;
	}
if ((DEBUG & DEBUG_IIS) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.read: " + amount);
return amount;
}

/**	Determine if the source data is cached.
<p>
	@return	true
*/
public boolean isCached ()
{return true;}

/**	Determine if the source data is cached in memory.
<p>
	@return	true
*/
public boolean isCachedMemory ()
{return true;}

/**	Get the size of the source file.
<p>
	@return	The size of the source file. This will be -1 if the size
		is not yet known. This information may become available after
		further interaction with the server providing access to the
		source file.
*/
public long length ()
{
if (Source_Size == NO_SOURCE_SIZE)
	return -1;
return Source_Size;
}

/**	Discard buffered data before a source location.
<p>
	@param	location	The source file offset before which buffered
		data is to be discarded. If the location is before the beginning
		of buffered data nothing is done.
	@throws	IOException If the source file stream is closed.
	@throws	IndexOutOfBoundsException If the location is greater than
		the current {@link #getStreamPosition() source input location}.
	@see	#Drain(int)
*/
public void flushBefore
	(
	long	location
	)
	throws IOException
{
if ((DEBUG & DEBUG_IIS) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.flushBefore: " + location);
if (location > Stream_Location ())
	throw new IndexOutOfBoundsException (ID + '\n'
		+ "Flush location " + location
			+ " is beyond the current stream location of "
			+ Stream_Location () + '.');
if (location > Buffer_Location ())
	Drain (Position_for_Location (location));
if ((DEBUG & DEBUG_IIS) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.flushBefore");
}

/**	Set the current source file location where the next byte will be read.
<p>
	The {@link #getBitOffset() bit offset} is set to zero.
<p>
	@param	location	The source file offset for the image buffer's
		next read position. A location less than zero will be set to zero;
		a location greater than the source file size will be set to the
		end of the source file.
	@throws	IOException If the source file stream is closed or there was
		a problem reading data from from the source.
	@see	#Stream_Location(long)
*/
public void seek
	(
	long	location
	)
	throws IOException
{
if ((DEBUG & DEBUG_IIS) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.seek: " + location);
Stream_Location (location);
bitOffset = 0;
if ((DEBUG & DEBUG_IIS) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.seek");
}

/**	All access to the source is closed.
<p>
	@throws	IOException If there was a problem closing the server
		input or output streams or the socket connection.
*/
public void close ()
	throws IOException
{
if ((DEBUG & DEBUG_IIS) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.close");
String
	message = null;
if (Server_Input != null)
	{
	try {Server_Input.close ();}
	catch (IOException exception)
		{
		message = ID + '\n'
			+ "Unable to close the server input stream.\n"
			+ exception.getMessage ();
		}
	}
if (Server_Writer != null)
	{
	try {Server_Writer.close ();}
	catch (IOException exception)
		{
		if (message == null)
			message = ID;
		message += "\n"
			+ "Unable to close the server output stream.\n"
			+ exception.getMessage ();
		}
	}
if (Server_Socket != null)
	{
	try {Server_Socket.setSoLinger (false, 0);}
	catch (SocketException exception) {}
	try {Server_Socket.close ();}
	catch (IOException exception)
		{
		if (message == null)
			message = ID;
		message += "\n"
			+ "Unable to close the server socket connection.\n"
			+ exception.getMessage ();
		}
	}
Server_Socket = null;
Server_Writer = null;
Server_Input = null;
Source_Size = NO_SOURCE_SIZE;
System.gc ();
if (message != null)
	throw new IOException (message);
if ((DEBUG & DEBUG_IIS) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.close");
}

/**	Checks if the server connection is closed.
<p>
	@return	true if the server connection is closed; false otherwise.
*/
public boolean is_Closed ()
{return Server_Socket == null;}

/*==============================================================================
	HTTP Client
*/
/**	Connect to the server.
<p>
	The specified URL checked for validity. If the URL does not specify a
	port number, the {@link #DEFAULT_PORT} will be used.
<p>
	<b>N.B.</b>: Establishing a new server connection causes any existing
	connection to be closed and the source input location to be {@link
	#Reset(long) Reset} to the beginning of the file.
<p>
	@param	url	The URL to use in connection to the server.
	@throws	IllegalArgumentException If the URL does not specify the
		http protocol nor provide a source file pathname.
	@throws	UnknownHostException If the hostname of the URL can not
		be resolved to an internet address.
	@throws	IOException If the socket connection could not be established
		or the input or output streams could not be opened.
	@see	#Reconnect()
*/
public void Connect
	(
	URL		url
	)
	throws IOException
{
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.Connect: " + url);

//	Protocol.
if (! "http".equals (url.getProtocol ()))
	{
	try {close ();}
	catch (IOException e) {}
	throw new IllegalArgumentException (ID + '\n'
		+ "Invalid URL: " + url + '\n'
		+ "Only the http protocol can be used.");
	}

//	Server hostname.
String
	server_hostname = null;
InetAddress
	address = null;
try
	{
	server_hostname = url.getHost ();
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("    Host name: " + server_hostname);
	address = InetAddress.getByName (server_hostname);
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("    Host address: " + address);
	}
catch (Exception exception)
	{
	try {close ();}
	catch (IOException e) {}
	throw new UnknownHostException (ID + '\n'
		+ "Invalid URL: " + url + '\n'
		+ "Unknown hostname.");
	}

//	Port number.
int
	port = url.getPort ();
if (port < 0)
	port = DEFAULT_PORT;
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		("    Port number: " + port);

//	Image pathname.
String
	source_pathname = url.getPath ();
if (source_pathname.length () == 0 ||
	source_pathname.equals ("/"))
	{
	try {close ();}
	catch (IOException e) {}
	throw new IllegalArgumentException (ID + '\n'
		+ "Invalid URL: " + url + '\n'
		+ "A source image pathname must be specified.");
	}
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		("    Image pathname: " + source_pathname);

//	Server internet socket address.
InetSocketAddress
	server_address = Server_Address;
Server_Address = new InetSocketAddress (address, port);

//	Connect (or reconnect) to the server.
try {Reconnect ();}
catch (IOException exception)
	{
	//	Try to restore the previous server connection.
	Server_Address = server_address;
	try {Reconnect ();}
	catch (IOException except) {}
	throw exception;
	}

//	Update the server URL connection information.
Source_URL = new URL
	(
	url.getProtocol (),
	Server_Hostname = server_hostname,
	port,
	Source_Pathname = source_pathname
	);

if (Source_Buffer != null)
	Reset (0);

if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.Connect");
}

/**	Reconnect to the server.
<p>
	If a Socket for communication with the server exists it is closed.
<p>
	A new Socket connection to the server is constructed. A maximum of
	{@link #DEFAULT_CONNECT_TIMEOUT} milliseconds is allowed for the
	connection to be established. Then input and output communication
	streams with the server are opened over the Socket. Up to {@link
	#MAX_RECONNECT_ATTEMPTS} connection attempts will be made. On the
	last attempt, the first connection operation failure that occurs will
	throw an IOException.
<p>
	@throws	IOException If a Socket could not be constructed on the
		server address within the {@link #DEFAULT_CONNECT_TIMEOUT} time,
		a Writer could not be constructed on the Socket or an InputStream
		could not be constructed on the Socket.
*/
protected void Reconnect ()
	throws IOException
{
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.Reconnect");
int
	attempt = MAX_RECONNECT_ATTEMPTS;
while (attempt-- != 0)
	{
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("    Attempt " + (MAX_RECONNECT_ATTEMPTS - attempt));
	//	Close the current socket.
	if (Server_Socket != null)
		{
		if (Logging)
			System.out.println (LOG_OUTPUT_PREFIX
				+ "[Server reconnect]");
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("    Closing the current connection");
		try {close ();}
		catch (IOException exception) {}
		/*
		//	Let the dust settle.
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("    Pausing "+ RECONNECT_WAIT + " ms");
		try {this.wait (RECONNECT_WAIT);}
		catch (InterruptedException exception) {}
		*/
		}

	if (Server_Address == null)
		//	No addres for reconnection.
		break;

	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("    Server Socket.");
	Server_Socket = new Socket ();
	try {Server_Socket.connect (Server_Address, DEFAULT_CONNECT_TIMEOUT);}
	catch (SocketTimeoutException exception)
		{
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("    Socket connect timeout.");
		try {close ();}
		catch (IOException except) {}
		if (attempt == 0)
			throw new IOException (ID + '\n'
				+ "Unable to connect to server at " + Server_Address + '\n'
				+ "after waiting " + (DEFAULT_CONNECT_TIMEOUT / 1000)
					+ " seconds.\n"
				+ exception.getMessage ());
		continue;
		}
	catch (IOException exception)
		{
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("    IOException: "+ exception.getMessage ());
		try {close ();}
		catch (IOException except) {}
		if (attempt == 0)
			throw new IOException (ID + '\n'
				+ "Unable to make a socket connection to the server at "
					+ Server_Address + '\n'
				+ exception.getMessage ());
		continue;
		}

	//	Output stream.
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("    Server Writer.");
	try
		{
		Server_Writer = new BufferedWriter
			(new OutputStreamWriter
				(Server_Socket.getOutputStream (), ENCODING_SCHEME),
					HEADERS_BUFFER_SIZE);
		}
	catch (IOException exception)
		{
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("    IOException: "+ exception.getMessage ());
		try {close ();}
		catch (IOException e) {}
		if (attempt == 0)
			throw new IOException (ID + '\n'
				+ "Unable to open an output stream to the server at "
					+ Server_Address + '\n'
				+ exception.getMessage ());
		continue;
		}

	//	Input stream.
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("    Server Input.");
	try {Server_Input = Server_Socket.getInputStream ();}
	catch (IOException exception)
		{
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("    IOException: "+ exception.getMessage ());
		try {close ();}
		catch (IOException e) {}
		if (attempt == 0)
			throw new IOException (ID + '\n'
				+ "Unable to open an input stream from the server at "
					+ Server_Address + '\n'
				+ exception.getMessage ());
		continue;
		}
	break;
	}
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.Reconnect");
}

/**	Redirect the source URL.
<p>
	@return	true if the HTTP Status indicated a redirection (300-399)
		and the redirection succeeded; false otherwise.
*/
private boolean Redirect ()
	throws IOException
{
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.Redirect");
boolean
	redirected = false;
if (HTTP_Status >= 300 &&
	HTTP_Status <  400)
	{
	String
		source_location = Header_Content ("Location");
	if (source_location != null)
		{
		Connect (new URL (source_location));
		redirected = true;
		}
	}
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.Redirect: " + redirected);
return redirected;
}

/*------------------------------------------------------------------------------
	Buffer Management
*/
/**	Source data buffer.
<p>
	Two views of the same data are provided: A Source_Buffer and an
	Image_Buffer.
<p>
	The Source_Buffer is filled (put) by the Server_Input socket stream.
	It's position is where the next data content - not header lines -
	input byte will be placed. It's limit is the buffer capacity.
<p>
	The Image_Buffer is used (get) to implement the ImageInputStream
	interface. It's position is where the next ImageInputStream read byte
	will be obtained; this is equivalent to the virtual stream location
	in the {@link #Source_Pathname()} on the {@link #Server_Hostname()}.
	It's limit is the end of valid source data in the buffer (i.e. the
	buffer may not be full). This should be the same as the Source_Buffer
	position.
<p>
	The first byte of the buffer is located at Buffer_Location offset in
	the source file. The streamPos variable in the ImageInputStreamImpl
	base class must also be maintained as the offset in the source file
	corresponding to the Image_Buffer position.
*/
private ByteBuffer
	Source_Buffer			= null,
	Image_Buffer;

//	Requirement: INPUT_INCREMENT < Source_Buffer.capacity
/**	The amount of source data to {@link #Fetch(int) fetch} when
	performing an automatic refresh of buffer contents.
*/
public static final int
	INPUT_INCREMENT			= 0x2000;				//   8 KB.

/**	The default buffer capacity to {@link #Allocate_Buffer(int) allocate}.
*/
public static final int
	DEFAULT_BUFFER_SIZE		= INPUT_INCREMENT << 4;	// 128 KB.


/**	Allocates a data storage buffer.
<p>
	A ByteBuffer of the specified size (bytes) is allocated and cleared
	(position set to zero and limit set to the capacity).
<p>
	If there is no previously existing source buffer, the newly allocated
	buffer is set as the source buffer. It is duplicated to provide the
	image buffer view and the image buffer's limit is set to zero.
<p>
	If there is a previously existing source buffer, all valid source
	data is copied from it into the new buffer, up to the capacity of the
	new buffer, and then the new buffer is set as the source buffer.
	<b>N.B.</b>: If the new buffer capacity is less than the previous
	buffer's valid data content, the excess data will be dropped. The
	source buffer is duplicated onto the temporary buffer, the image
	buffer limit and position copied over, and then the temporary buffer
	is set as the image buffer. <b>N.B.</b>: If the previous image buffer
	limit is greater than the new buffer capacity the new limit will be
	the new capacity; the same applies to the image buffer position,
	which would result in a corresponding move back of the logical stream
	location. Garbage collection is initiated to recover the previous
	buffer resources.
<p>
	@param	capacity	The capacity of the new buffer. If less than
		twice the {@link #INPUT_INCREMENT} it will be increased to this
		value.
*/
public void Allocate_Buffer
	(
	int		capacity
	)
{
if (capacity < (2 * INPUT_INCREMENT))
	capacity =  2 * INPUT_INCREMENT;
if ((DEBUG & DEBUG_BUFFERS) != 0)
	System.out.println
		(">>> HTTP_ImageInput_Stream.Allocate_Buffer: " + capacity);
ByteBuffer
	buffer = ByteBuffer.allocate (capacity);
buffer.clear ();	// position = 0; limit = capacity.

if (Source_Buffer == null)
	{
	Source_Buffer = buffer;
	Image_Buffer = Source_Buffer.duplicate ();
	Image_Buffer.limit (0);
	}
else
	{
	//	Copy over the existing data.
	if ((DEBUG & DEBUG_BUFFERS) != 0)
		System.out.println
			("    Copying " + Source_Buffer.position () + " data buffer bytes");
	Source_Buffer.flip ();
	if (Source_Buffer.limit () > buffer.capacity ())
		Source_Buffer.limit (buffer.capacity ());
	buffer.put (Source_Buffer);
	Source_Buffer = buffer;

	//	Replicate the image buffer view.
	buffer = Source_Buffer.duplicate ();
	if (Image_Buffer.position () > buffer.capacity ())
		{
		buffer.position (buffer.capacity ());
		streamPos = Location_for_Position (buffer.capacity ());
		}
	else
		buffer.position (Image_Buffer.position ());
	if (Image_Buffer.limit () > buffer.capacity ())
		buffer.limit (buffer.capacity ());
	else
		buffer.limit (Image_Buffer.limit ());
	Image_Buffer = buffer;

	//	Free unused memory.
	System.gc ();
	}
if ((DEBUG & DEBUG_BUFFERS) != 0)
	System.out.println
		("<<< HTTP_ImageInput_Stream.Allocate_Buffer");
}

/**	Drain data from the buffer.
<p>
	The logical location of the buffer in the source data stream is moved
	forward by some amount. This has the effect of draining valid source
	data from the front of the buffer and sliding the buffer forward
	to a new location in the source file.
<p>
	If the amount is less than or equal to zero, nothing is done.
<p>
	If the amount is greater than or equal to the end of valid source
	data in the buffer (image buffer limit), the buffer is simply (@link
	#Empty() emptied}.
<p>
	The remaining source data beyond the amount buffer index is moved to
	the beginning of the buffer. The buffer's logical position is the
	source data stream is moved forward by the amount of data removed
	(the slide distance). The source buffer input position and image
	buffer limit are reset to the new end of valid source data. The image
	buffer position is moved back by the amount of of data removed.
	However, if this new position would be less than zero it is set to
	zero, and the current logical stream location is moved up by the
	difference (i.e. to the buffer's logical position in the stream).
<p>
	@param	amount	The amount of data to drain.
	@return	The amount of source data effectively removed from the buffer
		(i.e. the slide distance).
*/
public int Drain
	(
	int		amount
	)
{
if (amount <= 0)
	return 0;
if ((DEBUG & DEBUG_BUFFERS) != 0)
	System.out.println (">>> HTTP_ImageInputStream.Drain: " + amount);

/*	N.B.: Image_Buffer limit == Source_Buffer position
	And they must remain the same.
*/
if (amount >= Image_Buffer.limit ())
	{
	//	Empty the buffer.
	if ((DEBUG & DEBUG_BUFFERS) != 0)
		System.out.println
			("<<< HTTP_ImageInputStream.Drain: emptied");
	return Empty ();
	}

//	Post-move location in the source data stream of the buffer.
Buffer_Location (Buffer_Location () + amount);
//	Post-move current stream location buffer index.
int
	position = Image_Buffer.position () - amount;
if (position < 0)
	{
	position = 0;
	/*
		Data is being drained beyond the current stream location.
		The stream location must be shifted up accordingly.
	*/
	streamPos = Buffer_Location ();
	}
//	Post-move end of valid source data buffer index.
int
	limit = Image_Buffer.limit () - amount;

//	Move the data.
if ((DEBUG & DEBUG_BUFFERS) != 0)
	System.out.println
		("    Moving data buffer bytes ["
			+ amount + '-' + Source_Buffer.position ()
			+ ") to [0-" + Source_Buffer.position () + ')');
Source_Buffer.limit (Source_Buffer.position ());	// End of data.
Source_Buffer.position (amount);					// Start of data.
Image_Buffer.position (0);							// Destination.
Image_Buffer.put (Source_Buffer);					// Move the data.

//	Reset the pointers.
Image_Buffer.position (position);
Image_Buffer.limit (limit);
Source_Buffer.position (limit);
Source_Buffer.limit (Source_Buffer.capacity ());
if ((DEBUG & DEBUG_BUFFERS) != 0)
	System.out.println
		("    Source_Buffer.position = " + Source_Buffer.position () + '\n'
		+"    Source_Buffer.limit    = " + Source_Buffer.limit () + '\n'
		+"    Image_Buffer.position  = " + Image_Buffer.position () + '\n'
		+"    Image_Buffer.limit     = " + Image_Buffer.limit () + '\n'
		+"    Buffer_Location = " + Buffer_Location () + '\n'
		+"    streamPos       = " + streamPos + '\n'
		+"<<< HTTP_ImageInputStream.Drain: " + amount);
return amount;
}

/**	Drain source data before the current stream position.
<p>
	This has the same effect as {@link #Drain(int) drain}ing all data
	before the current stream (image buffer) position.
<p>
	@return	The amount of source data effectively removed from the buffer
		(i.e. the slide distance).
	@see	#Drain(int)
*/
public int Drain ()
{return Drain (Image_Buffer.position ());}

/**	Empty the buffer of all source data.
<p>
	The buffer's logical location in the source data stream is moved up
	to the end of source data content in the buffer, and the current
	source data stream position is set to this location. The source
	buffer position is set to the beginning of the buffer as are the
	image buffer position and data limit. This has the same effect as
	{@link #Drain(int) drain}ing all the data content from the buffer.
<p>
	@return	The amount of source data effectively removed from the buffer
		(i.e. the slide distance).
*/
public int Empty ()
{
int
	amount = Image_Buffer.limit ();
if (amount > 0)
	Reset (Data_End_Location ());
return amount;
}

/**	Reset the buffer to a new source location and empty it.
<p>
	<b.N.B.</b>: All buffered source data is dropped regardless of
	the location relative to the current buffered source data range.
<p>
	@param	location	The new buffer and stream position location.
*/
public void Reset
	(
	long	location
	)
{
Buffer_Location =
streamPos = location;
Source_Buffer.position (0);
Image_Buffer.limit (0);
}

/**	Get the virtual source location of the beginning of the buffer.
<p>
	@return	The source file offset where the buffer begins.
*/
public long Buffer_Location ()
{return Buffer_Location;}

/**	Set the virtual source location of the beginning of the buffer.
<p>
	<b>N.B.</b>: Changing the virtual source location of the buffer will
	affect all other buffer virtual locations because they are relative
	to the buffer location.
<p>
	@param	location	The source file offset where the buffer begins.
		A value less than zero is set to zero; a value greater than the
		size of the source file is set to the size of the source file.
*/
protected void Buffer_Location
	(
	long	location
	)
{
if (location < 0)
	location = 0;
else
if (location > Source_Size)
	location = Source_Size;
Buffer_Location = location;
}

/**	Get the virtual source location where the next image buffer will be
	read.
<p>
	@return	The current image buffer virtual source file offset location.
*/
protected long Stream_Location ()
{return Location_for_Position (Image_Buffer.position ());}

/**	Set the virtual source location where the next image buffer will be
	read.
<p>
	If the location is within the range of buffered source data the
	image buffer position is simply moved to the corresponding position.
<p>
	The location is less than zero, zero will be used; if it is greater
	than the size of the source file, the location will be set to the
	end of the source data.
<p>
	If the location is the same as the current virtual source stream
	location nothing is done.
<p>
	If the location is before the stream location of the first byte in
	the image buffer and within {@link #INPUT_INCREMENT} bytes of the
	beginning of the buffer, the current buffered data will be shifted
	up by INPUT_INCREMENT bytes and the resulting gap at the beginning
	of the buffer filled with data {@link #Fetch(int) Fetch}ed from the
	corresponding offset range of the source file. The image buffer
	position is then set to correspond to the new stream location.
<p>
	If the location is beyond the end of buffered data, but not beyond
	the end of the maximum capacity of the buffer, and within
	INPUT_INCREMENT bytes of the end of data, up to INPUT_INCREMENT
	additional bytes (not more than would fill the buffer to capacity)
	are {@link #Fetch(int) Fetch}ed from the source file to extend the
	buffered data. The image buffer position is then set to correspond
	to the new stream location.
<p>
	For any other case the buffer is simply {@link #Reset(long) Reset}
	to the new location.
<p>
	The effect is a random access seek to any location within the
	source file, with data pre-fetch optimization when the new location
	is near the currently buffered data.
<p>
	@param	location	The source file offset for the image buffer's
		next read position.
*/
public void Stream_Location
	(
	long	location
	)
	throws IOException
{
if ((DEBUG & DEBUG_BUFFERS) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.Stream_Location: " + location);
if (location >= Buffer_Location () &&
	location <= Data_End_Location ())
	{
	//	The new location is within the available buffered data.
	Image_Buffer.position ((int)(location - Buffer_Location ()));
	streamPos = location;
	if ((DEBUG & DEBUG_BUFFERS) != 0)
		System.out.println
			("    The location is within the buffer.\n"
			+"<<< HTTP_ImageInputStream.Stream_Location: " + streamPos);
	return;
	}

if (location < 0)
	location = 0;
else
if (location > Source_Size)
	location = Source_Size;
if (location == streamPos)
	{
	if ((DEBUG & DEBUG_BUFFERS) != 0)
		System.out.println
			("    The location is unmoved.\n"
			+"<<< HTTP_ImageInputStream.Stream_Location: " + streamPos);
	return;
	}

if (location < Buffer_Location ())
	{
	//	Move the logical stream location back before the buffer location.
	if ((Buffer_Location () - location) <= INPUT_INCREMENT)
		{
		/*	The seek distance is no greater than the input increment.

			Move data an input increment to prevent inefficient "creep".
			Make a hole for the input data by shifting the existing data
			up so that it will be contiguous with the new data that is
			read into the hole.

			Data is moved from the upper end to prevent data overlap.
		*/
		int
			source = Image_Buffer.limit () - 1,	//	Source byte index.
			target = source + INPUT_INCREMENT;	//	Destination index.
		if (target >= Image_Buffer.capacity ())
			{
			//	Clip off the excess data.
			source -= target - Image_Buffer.capacity () + 1;
			target = Image_Buffer.capacity () - 1;
			}
		Image_Buffer.limit (target + 1);	//	New end of data index.

		//	Move the data.
		if ((DEBUG & DEBUG_BUFFERS) != 0)
			System.out.println
				("    Moving data buffer bytes [0-" + (source + 1)
					+ ") to [" + (Image_Buffer.limit () - source)
					+ '-' + (target + 1) + ')');
		while (source >= 0)
			Image_Buffer.put (target--, Image_Buffer.get (source--));

		//	Fill the hole with an input increment of source data.
		target = Image_Buffer.limit ();		//	Save end of data position.
		Reset (Buffer_Location () - INPUT_INCREMENT);
		Fetch (INPUT_INCREMENT);
		//	Reset the buffer pointers.
		Source_Buffer.position (target);	//	Restore end of data position.
		Image_Buffer.limit (target);
		Image_Buffer.position (Position_for_Location (location));
		streamPos = location;
		}
	else
		//	The seek distance is larger than the input increment.
		Reset (location);
	}
else
	{
	//	Move the logical location forward beyond the end of data.
	if (location < Location_for_Position (Image_Buffer.capacity ()) &&
		(location - INPUT_INCREMENT) < Data_End_Location ())
		{
		//	The new location is within an INPUT_INCREMENT of end of data.
		int
			amount = INPUT_INCREMENT;
		if ((Image_Buffer.limit () + amount) > Image_Buffer.capacity ())
			amount = Image_Buffer.capacity () - Image_Buffer.limit ();
		Fetch (amount);
		Image_Buffer.position (Position_for_Location (location));
		streamPos = location;
		}
	else
		//	The seek distance is larger than the input increment.
		Reset (location);
	}
if ((DEBUG & DEBUG_BUFFERS) != 0)
	System.out.println
		("    Source_Buffer.position = " + Source_Buffer.position () + '\n'
		+"    Source_Buffer.limit    = " + Source_Buffer.limit () + '\n'
		+"    Image_Buffer.position  = " + Image_Buffer.position () + '\n'
		+"    Image_Buffer.limit     = " + Image_Buffer.limit () + '\n'
		+"    Buffer_Location = " + Buffer_Location () + '\n'
		+"    streamPos       = " + streamPos + '\n'
		+"<<< HTTP_ImageInputStream.Stream_Location: " + streamPos);
}

/**	Get the virtual source location of the end (exclusive) of buffered
	data.
<p>
	@return	The source file offset where buffered data ends.
*/
public long Data_End_Location ()
{return Location_for_Position (Image_Buffer.limit ());}

/**	Set the virtual source location of the end (exclusive) of buffered
	data.
<p>
	<b>N.B.</b>: If the new location is less than the buffer's current
	virtual source location, the virtual source location will also be
	moved to the new location. This enforces the constraint that the
	virtual source location is always less than or equal to the end of
	data location.
<p>
	@param	location	The source file offset where the end of buffered
		data is located. The location is clipped to produce a buffer
		position with the range of the buffer capacity.
*/
protected void Data_End_Location
	(
	long	location
	)
{
int
	index = Position_for_Location (location);
if (index < 0)
	index = 0;
else
if (index > Image_Buffer.capacity ())
	index = Image_Buffer.capacity ();
Image_Buffer.limit (index);
Source_Buffer.position (index);
}

/**	Convert a buffer position to a virtual source location.
<p>
	@param	position	A buffer position int.
	@return	A source location long.
*/
protected long Location_for_Position
	(
	int		position
	)
{return Buffer_Location () + position;}

/**	Convert a virtual source location to a buffer position.
<p>
	@param	location	A source location long.
	@return	A buffer position int.
*/
protected int Position_for_Location
	(
	long	location
	)
{return (int)(location - Buffer_Location ());}

/**	Clip a source location to the currently buffered data range.
<p>
	@param	location	A source location long.
	@return	The source location clipped to the range of buffered data.
*/
public long Image_Buffer_Range
	(
	long	location
	)
{
if (location < Buffer_Location ())
	return Buffer_Location ();
if (location < Data_End_Location ())
	return location;
return Data_End_Location ();
}

/**	Test if a source location is in the range of buffered data.
<p>
	The buffered data range begins at the {@link #Buffer_Location()}
	(inclusive) and extends to the {@link #Data_End_Location()} (exclusive).
<p>
	@param	location	A source location long.
	@return	true if the location is the range of buffered data; false
		otherwise.
*/
public boolean in_Image_Buffer_Range
	(
	long	location
	)
{
return
	location >= Buffer_Location () &&
	location <  Data_End_Location ();
}

/*------------------------------------------------------------------------------
	Request
*/
/**	Check the server status of the source.
<p>
	A "HEAD" request for the source file is made to the server. The
	reponse to a "HEAD" request will contain all the HTTP headers of a
	normal request, but no data content will be included.
<p>
	If access to the server is closed, then the last HTTP status value
	will be returned. If no HTTP status value was ever obtained zero
	will be returned. If an end of file condition is encountered will
	communicating with the server, -1 will be returned. If some other
	IOException is thrown, -2 will be returned. If any other Exception
	is thrown, -3 will be returned.
<p>
	The HTTP status when the source file is correctly found is less than
	{@link #SUCCESS_STATUS_LIMIT}. If the source file is not found the
	status will be {@link #SOURCE_NOT_FOUND_STATUS}.
<p>
	@return	The HTTP response status.
*/
public int Check_Source ()
{
if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.Check_Source");
if (! is_Closed ())
	{
	try
		{
		Request ("HEAD", Source_Pathname);
		End_Request ();
		Headers ();
		}
	catch (Exception exception)
		{
		HTTP_Status =
			((exception instanceof EOFException) ? -1 :
			((exception instanceof IOException)  ? -2 : -3
			));
		}
	}
if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.Check_Source: " + HTTP_Status);
return HTTP_Status;
}

/**	Fetch source data from the server (absolute fetch).
<p>
	The data will be fetched from the source file beginning at the start
	location up to, but not including, the end location.
<p>
	If the start location is within the buffered source data range it is
	moved up to the end of data location. If the start location is at or
	beyond the end of the source file, nothing is done and zero is
	returned. If, after range clipping, the start location is not the
	same as the end of source data location in the buffer, the buffer is
	is {@link #Reset(long) reset} to the start location. The end
	location will be clipped to the size of the source file.
<p>
	If communication with the server is lost during the data fetch, a
	{@link #Reconnect() reconnect}ion will be tried. If this succeeds
	the data fetch will be repeated.
<p>
	@param	start	The source file offset (inclusive) where the data
		fetch is to begin.
	@param	end		The source file offset (exclusive) where the data
		fetch is to end.
	@return	The amount of data received from the server.
	@throws	IllegalArgumentException If start is less than zero or
		greater than end.
	@throws	IOException If there was a problem communicating with the
		server.
	@see	#Request_Source_Data(long, long)
	@see	#Receive()
*/
public int Fetch
	(
	long	start,
	long	end
	)
	throws IllegalArgumentException, IOException
{
if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.Fetch: " + start + '-' + end);
if (Server_Writer == null)
	throw new IOException (ID + '\n'
		+ "The server connection is closed.");
int
	received = 0;

if (start < 0 ||
	start > end)
	throw new IllegalArgumentException (ID + '\n'
		+ "Invalid source data range to fetch: " + start + '-' + end);
if (in_Image_Buffer_Range (start))
	start = Data_End_Location ();
if (start < Source_Size)
	{
	if (end > Source_Size)
		end = Source_Size;

	if (start != Data_End_Location ())
		//	Reset the buffer to the new start location.
		Reset (start);
	Request_Source_Data (start, end);
	received = Receive ();
	if (received < 0)
		{
		try
			{
			Reconnect ();
			Request_Source_Data (start, end);
			received = Receive ();
			if (received < 0)
				{
				throw new IOException (ID + '\n'
					+ "Sever connection lost while fetching from "
						+ start + " to " + end);
				}
			}
		catch (IOException exception)
			{
			throw new IOException (exception.getMessage () + '\n'
				+ "Sever connection lost while fetching from "
					+ start + " to " + end);
			}
		}
	}
if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.Fetch: " + received);
return received;
}

/**	Fetch source data from the server (relative fetch).
<p>
	The data will be fetched from the source file starting at the
	current end of source data location in the buffer.
<p>
	@param	amount	The amount of data to fetch.
	@return	The amount of data fetched.
	@throws	IOException	If there was a problem send or receiving data
		from the server.
*/
public int Fetch
	(
	int		amount
	)
	throws IOException
{return Fetch (Data_End_Location (), Data_End_Location () + amount);}

/**	Get a range of source data.
<p>
	<b>N.B.</b>: No check is made of the validity of the source range.
<p>
	@param	start	The file offset of the start of the data range
		(inclusive).
	@param	end		The file offset of the end of the data range
		(exclusive).
	@return	This HTTP_ImageInputStream.
	@throws	IllegalArgumentException If start is greater than end.
	@throws	BufferOverflowException If the (end - start) amount is
		greater than Integer.MAX_VALUE.
	@throws	IOException If there was a problem sending the request
		to the server.
*/
protected HTTP_ImageInputStream Request_Source_Data
	(
	long	start,
	long	end
	)
	throws IllegalArgumentException, BufferOverflowException, IOException
{
if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.Request_Source_Data: " + start + "-" + end);
if (start > end)
	throw new IllegalArgumentException (ID + '\n'
		+ "Invalid source data range request: " + start + "-" + end);
if ((end - start) > (long)Integer.MAX_VALUE)
	throw new BufferOverflowException ();
if (start < end)
	{
	Request ("GET", Source_Pathname);
	//	N.B.: Request range limits are inclusive.
	Send ("Range: bytes=" + start + "-" + --end);
	Send ("Accept-Encoding: identity");
	End_Request ();
	}
if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.Request_Source_Data");
return this;
}

/**	Issue a Request preamble to the server.
<p>
	After the HTTP formatted request is {@link #Send(String) sent} to
	the server, the Host, User-Agent, and Cache-Control (no-cache)
	requests are also sent. <b>N.B.</b>: The request sequence has not
	been {@link #End_Request() ended}.
<p>
	@param	method	The request method String. This is typically "GET".
	@param	request	The request operation.
	@see	#Send(String)
*/
protected void Request
	(
	String	method,
	String	request
	)
	throws IOException
{
if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		(">>> HTTP_ImageInputStream.Request: " + method + ' ' + request);
Send (method + ' ' + request + ' ' + HTTP_VERSION);
Send ("Host: " + Server_Hostname + ':' + Server_Address.getPort ());
Send ("User-Agent: " + User_Agent);
Send ("Cache-Control: no-cache");
if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.Request");
}

/**	Send the end-of-request sequence to the server.
<p>
	An single empty line is sent.
*/
protected void End_Request ()
	throws IOException
{Send ("");}

/**	Send a message line to the server.
<p>
	The message is expected to be a single line to be sent to the
	server. An end of line sequence will be appended to the message that is
	sent if it is not already present.
<p>
	If {@link #Logging(boolean) Logging} is enabled the message,
	preceded by the {@link #LOG_OUTPUT_PREFIX}, will be printed to
	stdout before it is sent to the server.
<p>
	@param	message	The message String to be sent.
	@throws	IOException If the connection to the server is closed or
		there was a problem sending the message.
*/
protected void Send
	(
	String	message
	)
	throws IOException
{
if (is_Closed ())
	throw new IOException (ID + '\n'
		+ "The server connection is closed.");
if (message.length () == 0 ||
	message.charAt (message.length () - 1) != '\n')
	message += EOL;
Server_Writer.write (message);
Server_Writer.flush ();
Total_Sent += message.length ();
if (Logging)
	System.out.print (LOG_OUTPUT_PREFIX + message);
}

/*------------------------------------------------------------------------------
	Response
*/
/**	Receive a response to a server request.
<p>
	Server interaction always occurs in {@link #Request(String,String)
	Request}-Receive pairs. After a Request has been sent it is necessary
	to Receive the response from the server. The first part of the
	response is sequence of one or more HTTP header lines. If a
	Content-Length line is present the header lines are followed by data
	content of Content-Length bytes.
<p>
	If the HTTP response headers contain a {@link #Status() Status}
	value that is less than 300, the data content is read into the
	Source_Buffer starting at its current position. However, if the
	amount of free space remaining in the buffer (buffer storage not
	occupied by valid data
	content; i.e. locations [position, limit), where limit == capacity)
	is less than the Content-Length, more buffer space is first provided.
<p>
	Space to store data content in the buffer is provided by {@link
	#Drain() drain}ing away data from the beginning of the buffer
	before the current source stream position.
<p>
	@return	The amount of source data received, or -1 if EOF was
		encountered before any data was received.
*/
protected int Receive ()
	throws IOException
{
if ((DEBUG & DEBUG_RECEIVE) != 0)
	System.out.println (">>> HTTP_ImageInputStream.Receive");
//	Get the HTTP headers.
try {Headers ();}
catch (EOFException exception)
	{
	if ((DEBUG & DEBUG_RECEIVE) != 0)
		System.out.println ("<<< HTTP_ImageInputStream.Receive: -1");
	return -1;
	}

long
	content_length = Content_Length ();
if (content_length > Integer.MAX_VALUE)
	{
	/*	Too much data content!

		This should never happen. But if it does something is wrong
		and the Server_Input is now jammed with a huge queue of
		data that must be dumped. Instead of locking up the user
		application needlessly reading it all in, the connection to
		the server will be closed and a BufferOverflowException
		thrown.
	*/
	close ();
	throw new BufferOverflowException ();
	}

Range
	range = Content_Range ();
int
	amount,
	received = 0,
	length = (int)content_length;
if (length > 0)
	{
	if (Status () < 300)
		{
		if ((DEBUG & DEBUG_RECEIVE) != 0)
			System.out.println
				("    Response status " + Status () + '\n'
				+"    Content range:  " + range + '\n'
				+"    Content length: " + length);
		if (range.Start != Data_End_Location ())
			{
			if ((DEBUG & DEBUG_RECEIVE) != 0)
				System.out.println
					("    Content is not contiguous with buffer content.\n"
					+"    Buffer reset to location " + range.Start);
			Reset (range.Start);
			}
		if (length > Source_Buffer.remaining ())
			{
			if ((DEBUG & DEBUG_RECEIVE) != 0)
				System.out.println
					("    Only " + Source_Buffer.remaining ()
						+ " bytes of free buffer space remaining.");
			//	More buffer space needed. Drain the buffer.
			Drain ();

			if (length > Source_Buffer.remaining ())
				{
				/*	Even more buffer space needed.

					Allocate a larger buffer that is a multiple
					of the INPUT_INCREMENT.
				*/
				amount = length - Source_Buffer.remaining ();
				amount += Source_Buffer.capacity () + INPUT_INCREMENT
						- (amount % INPUT_INCREMENT);
				Allocate_Buffer (amount);
				}
			}

		while (received < length)
			{
			amount = length - received;
			if ((amount = Server_Input.read
					(Source_Buffer.array (), Source_Buffer.position (), amount))
				< 0)
				break;				

			if ((DEBUG & DEBUG_DATA) != 0)
				System.out.println
					("    Got " + amount + " -\n"
					+"-->|"
						+ new String_Buffer (new String (Source_Buffer.array (),
								Source_Buffer.position (), amount))
							.special_to_escape ()
						+ "|<--");
			Data_End_Location (Data_End_Location () + amount);
			received += amount;
			Total_Received += amount;
			}
		if (Logging)
			System.out.println (LOG_INPUT_PREFIX +
				"Received " + received + " of " + length + " bytes.");
		}
	else
		{
		if ((DEBUG & DEBUG_RECEIVE) != 0)
			System.out.println
				("    Response status " + Status () + '\n'
				+"    Skipping " + length + " bytes.");
		Server_Input.skip (length);
		}
	}
if ((DEBUG & DEBUG_RECEIVE) != 0)
	System.out.println ("<<< HTTP_ImageInputStream.Receive: " + received);
return received;
}

/*------------------------------------------------------------------------------
	HTTP Response Headers
*/
/**	Get the HTTP response headers from the server.
<p>
	After a {@link #Request(String, String) Request} has been sent to the
	server - probably followed by {@link #Send(String) Send}ing
	additional request messages - and {@link #End_Request() End}ed, the
	server response will begin with a sequence of HTTP headers. Each
	response line up to, but not including, an empty line will be added
	to a Vector of HTTP Headers in which each element is itself a Vector
	containing the header name and content. Note: A header line can be
	wrapped across multiple response lines; the wrapped line(s) begin
	with whitespace. In addition, a header with the same name as a
	previous header will have its content appended to previous header.
	This guarantees that each header in the Vector of HTTP Headers will
	have a unique name.
<p>
	Before the header lines are accumulated the HTTP Headers Vector is
	cleared and the HTTP response {@link #Status() Status} is reset to
	zero. As the header lines are being accumulated the HTTP response
	status will be updated if it is found. In addition, the {@link
	#Source_Size() source file size} will also be set if it is found the
	response.
<p>
	@return	The HTTP response status.
	@throws	IOException	If an problem was encountered while reading
		response lines. This will be an EOFException if an end of
		input from the server was encountered.
*/
protected int Headers ()
	throws IOException
{
if ((DEBUG & DEBUG_HEADERS) != 0)
	System.out.println (">>> HTTP_ImageInputStream.Headers");
//	Reset for new headers from the server.
HTTP_Headers.clear ();
HTTP_Status = 0;

String
	line,
	header = "";
while (true)
	{
	//	Get a line from the server response.
	line = Response_Line ();

	if (line.length () == 0 ||
		WHITESPACE.indexOf (line.charAt (0)) < 0)
		{
		//	Start of a new header. Add the previous header.
		Header_Add (header);

		if (line.length () == 0)
			//	Last line.
			break;
		//	Reset the header to the current line.
		header = line;
		}
	else
		{
		//	Continuation of wrapped line.
		if (header.length () == 0)
			/*	The first line can't be wrapped.
				Set the header to the current line trimmed of whitespace.
			*/
			header = line.trim ();
		else
			//	Append this line to the header.
			header += line;
		}
	}

//	Attempt to update the HTTP response status.
Status ();
if (Source_Size == NO_SOURCE_SIZE)
	//	Attempt to set the source file size.
	Source_Size ();
if ((DEBUG & DEBUG_HEADERS) != 0)
	System.out.println ("<<< HTTP_ImageInputStream.Headers: " + HTTP_Status);
return HTTP_Status;
}

/**	A server response header line is read from the server.
<p>
	One byte at a time is read from the Server_Input until an end-of-line
	or end-of-file condition is encountered. All bytes up to, but not
	including, the end-of-line delimiter(s) are accumulated as characters
	in a StringBuffer.
<p>
	An end-of-line condition is identified by a '\n' or '\r' character. A
	'\r' may optionally be followed by a '\n' character which is treated
	as an additional end-of_line delimiter. The end-of-line characters
	are not included in the returned String. The Server_Input is left
	positioned at the next byte after the end-of-line delimiter(s).
<p>
	If an end-of-file condition is encountered after one or more line
	characters have been accumulated, a String is returned with those
	characters. However, if no characters have been accumulated an
	EOFException is thrown.
<p>
	@return	A String containing the line that was read.
	@throws	IOException if there was a problem reading from the
		Server_Input. This will ba an EOFException if an end-of-file
		condition was encountered before any line characters were read.
*/
private String Response_Line ()
	throws IOException
{
StringBuffer
	line = new StringBuffer ();
int
	datum;
Line:
while ((datum = Server_Input.read ()) >= 0)
	{
	++Total_Received;
	switch (datum)
		{
		case '\r':
			//	Probable NL after CR.
			Server_Input.mark (1);
			if ((datum = Server_Input.read ()) >= 0 &&
				datum != '\n')
				{
				//	Push the byte back onto the stream.
				Server_Input.reset ();
				--Total_Received;
				}
		case '\n':
			break Line;
	    default:
			line.append ((char)datum);
	    }
	}
if (datum < 0)
	{
	//	EOF
	if (line.length () == 0)
		throw new EOFException (ID + '\n'
			+ "Unexpected server disconnect.");
	}
if (Logging)
	System.out.println (LOG_INPUT_PREFIX + line);
return line.toString ();
}

/**	Add a field String to the HTTP Headers Vector.
<p>
	The first word of the field String is the header name; the remainder
	is the header content (this may be empty). If a header by the same
	name already exists in the HTTP Headers Vector, its content is
	{@link #Header_Append(String, String) appended} with the new
	content.
<p>
	@param	field_string	The HTTP header line String.
*/
private void Header_Add
	(
	String	field_string
	)
{
if (field_string == null ||
	field_string.length () == 0)
	return;
//	Break the field string into a field name-content vector.
Words
	words = new Words (field_string)
		.Delimiters (field_string.startsWith ("HTTP") ?
			HTTP_VERSION_DELIMITERS : FIELD_CONTENT_DELIMITERS);
Vector
	field = words.Split (2);
if (field.size () != 0)
	{
	if (field.size () == 1)
		//	Provide empty content.
		field.add ("");

	if (! Header_Append ((String)field.get (0), (String)field.get (1)))
		//	New field.
		HTTP_Headers.add (field);
	}
}

/**	Append header content to an existing header.
<p>
	If a header with the name exists in the HTTP Headers Vector
	it's content is appended, after a space character, with the
	content String.
<p>
	@param	name	The name of a header to have its content
		appended.
	@param	content	The content String to be appended.
	@return	true if a header with the name was found; false otherwise.
*/
private boolean Header_Append
	(
	String	name,
	String	content
	)
{
if (name == null ||
	content == null ||
	content.length () == 0)
	return false;
int
	index = Header_Index (name);
if (index < 0)
	return false;

Vector
	field = (Vector)HTTP_Headers.get (index);

/*	N.B.: It is assumed that the field will always have
	field name and field content elements,
	even if one or both are empty.
*/
field.set (1, (String)field.get (1) + ' ' + content);

return true;
}

/**	Get the HTTP Headers Vector index for a named header.
<p>
	@param	name	The name of a header.
	@return	The HTTP Headers Vector index for the named header, or -1
		if the header name was not found.
*/
private int	Header_Index
	(
	String	name
	)
{
if (name != null)
	{
	int
		index = 0,
		total = HTTP_Headers.size ();
	while (index < total)
		{
		/*	N.B.: It is assumed that the field will always have
			a field name element, even if it is empty.
		*/
		Vector
			field = (Vector)HTTP_Headers.get (index);
		if (name.equalsIgnoreCase
				((String)((Vector)HTTP_Headers.get (index)).get (0)))
			return index;
		++index;
		}
	}
return -1;
}

/**	Get the header content for a named header.
<p>
	@param	name	The name of a header.
	@return	The content String for the named header, or null if the
		header name was not found.
*/
public String Header_Content
	(
	String	name
	)
{
String
	content = null;
int
	index = Header_Index (name);
if (index >= 0)
	/*	N.B.: It is assumed that the field will always have
		field name and field content elements,
		even if one or both are empty.
	*/
	content = (String)((Vector)HTTP_Headers.get (index)).get (1);
return content;
}

/**	Get the complete header field for a named header.
<p>
	The header field is provided in the same syntax as it was
	received from the server. However, any header wrapping has been
	removed. If multiple headers with the same name were received,
	a single header field with all content appended in the order it
	was received is provided.
<p>
	@param	name	The name of a header.
	@return	The complete header field String.
*/
public String Header_Field
	(
	String	name
	)
{
String
	content = Header_Content (name);
if (content != null);
	name +=
		(name.equalsIgnoreCase ("HTTP") ?
			HTTP_VERSION_DELIMITERS.charAt (0) :
			FIELD_CONTENT_DELIMITERS.charAt (0)) +
		' ' + content;
return name;
}

/**	Get a header field value by name.
<p>
	A header field is divided into value fields by any one of the ':',
	';', ',', '&' or whitespace characters. A value field occurs in in
	header content as a value name separated from its value by an '='
	character.
<p>
	@param	name	The name of a header.
	@param	value_name	The name of a content value.
	@return	The value String. This will be null if the header or value
		name is not found.
*/
public String Header_Value
	(
	String	name,
	String	value_name
	)
{
if (value_name == null)
	return null;
String
	value = null,
	content = Header_Content (name);
if (content != null)
	{
	Vector
		values = new Words (content)
			.Delimiters (FIELD_CONTENT_DELIMITERS)
			.Split ();
	for (int index = 0,
				total = values.size ();
			 index < total;
		   ++index)
		{
		Vector
			field_value = new Words ((String)values.get (index))
				.Delimiters (VALUE_DELIMITERS)
				.Split (2);
		if (field_value.size () == 0)
			//	Empty value!
			continue;
		String
			key = (String)field_value.get (0);
		if (value_name.equalsIgnoreCase (key))
			{
			if (field_value.size () == 1)
				//	Only a value.
				value = key;
			else
				//	key=value.
				value = ((String)field_value.get (1)).trim ();
			break;
			}
		}
	}
return value;
}

/**	Get a header field value by index within the field content.
<p>
	@param	name	The name of a header.
	@param	value_index	An index int, starting with zero for the
		first value in the content.
	@return	The value String. This will be null if the header or value
		is not found.
	@see	#Header_Value(String, String)
*/
public String Header_Value
	(
	String	name,
	int		value_index
	)
{
if (value_index < 0)
	return null;
if ((DEBUG & DEBUG_HEADERS) != 0)
	System.out.println (">>> HTTP_ImageInputStream.Header_Value: "
		+ name + ' ' + value_index);
String
	value = null,
	content = Header_Content (name);
if (content != null)
	{
	Vector
		values = new Words (content)
			.Delimiters (FIELD_CONTENT_DELIMITERS)
			.Split ();
	if ((DEBUG & DEBUG_HEADERS) != 0)
		System.out.println ("    HTTP_ImageInputStream.Header_Value:\n"
			+ "    values: " + values);
	if (values.size () > value_index)
		{
		values = new Words ((String)values.get (value_index))
			.Delimiters (VALUE_DELIMITERS)
			.Split (2);
		if ((DEBUG & DEBUG_HEADERS) != 0)
			System.out.println
				("    value: " + values);
		if (values.size () == 0)
			//	Empty value!
			value = "";
		else if (values.size () == 1)
			//	Only a value.
			value = (String)values.get (0);
		else
			//	key=value.
			value = ((String)values.get (1)).trim ();
		}
	}
if ((DEBUG & DEBUG_HEADERS) != 0)
	System.out.println (">>> HTTP_ImageInputStream.Header_Value: " + value);
return value;
}

/**	Get the HTTP status of the last response from the server.
<p>
	The HTTP status is expected to be the second field value of the
	"HTTP" response header.
<p>
	@return	The HTTP status value. This will be zero if no HTTP status
		has yet been found. It will be negative if the expected HTTP
		status field does not contain an integer value.
*/
public int Status ()
{
if (HTTP_Status > 0)
	return HTTP_Status;
if ((DEBUG & DEBUG_HEADERS) != 0)
	System.out.println (">>> HTTP_ImageInputStream.Status");
String
	status_string = Header_Value ("HTTP", 1);
if (status_string != null)
	{
	try {HTTP_Status = Integer.parseInt (status_string);}
	catch (NumberFormatException exception)
		{HTTP_Status = -2;}
	}
if ((DEBUG & DEBUG_HEADERS) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.Status: " + HTTP_Status);
return HTTP_Status;
}

/**	Get the size of the source file.
<p>
	If the source file size has already been determined the cached value
	is returned.
<p>
	The size of the source file is obtained from the {@link
	#Content_Range() "Content-Range"} HTTP response header. If a valid
	size value is found the cached value is updated and that value
	returned.
<p>
	@return	The size of the source file. This will be {@link #NO_SOURCE_SIZE}
		if the size is unknown.
*/
public long Source_Size ()
{
if (Source_Size != NO_SOURCE_SIZE)
	return Source_Size;
if ((DEBUG & DEBUG_HEADERS) != 0)
	System.out.println (">>> HTTP_ImageInputStream.Source_Size");
Range
	range = Content_Range ();
if (range != null &&
	range.Length >= 0)
	Source_Size = range.Length;
if ((DEBUG & DEBUG_HEADERS) != 0)
	System.out.println
		("<<< HTTP_ImageInputStream.Source_Size: " + Source_Size);
return Source_Size;
}

/**	Get the length of server response data content.
<p>
	The length of the data content is expected to be the first field
	of the "Content-Length" HTTP response header.
<p>
	@return	The long value length of the response data content. This
		will be -1 if the content length could not be found. It will be
		-2 if the expected HTTP header field does not contain an valid
		value.
*/
public long Content_Length ()
{
if ((DEBUG & DEBUG_HEADERS) != 0)
	System.out.println (">>> HTTP_ImageInputStream.Content_Length");
long
	length = -1;
String
	length_string = Header_Value ("Content-Length", 0);
if (length_string != null)
	{
	try {length = Long.parseLong (length_string);}
	catch (NumberFormatException exception)
		{length = -2;}
	}
if ((DEBUG & DEBUG_HEADERS) != 0)
	System.out.println ("<<< HTTP_ImageInputStream.Content_Length: " + length);
return length;
}

/**	Get the Content-Range header values.
<p>
	The format of the Content-Range header is:
<p>
	Content-Range: bytes <i>start</i>-<i>end</i>/<i>length</i>
<p>
	Where <start> and <end> are byte offsets into the source file
	for the start and end (inclusive) of the bytes returned. The
	<length> value is the total length of the source file.
<p>
	@return	A Range containing the Content-Range values, or null
		if no Content-Range header could be found. Missing values
		in the range are indicated by the value -1.
*/
public Range Content_Range ()
{
if ((DEBUG & DEBUG_HEADERS) != 0)
	System.out.println (">>> HTTP_ImageInputStream.Content_Range");
Range
	range = null;
String
	range_string = Header_Value ("Content-Range", 1);
Values:
if (range_string != null)
	{
	range = new Range (-1, -1, -1);
	int
		first = 0,
		last;
	//	Start
	if ((last = range_string.indexOf ('-')) <= 0)
		break Values;
	try {range.Start = Long.parseLong (range_string.substring (first, last));}
	catch (NumberFormatException exception)
		{break Values;}

	//	End
	first = ++last;
	if ((last = range_string.indexOf ('/')) <= 0)
		{
		//	Try using the remainder of the string.
		if ((last = range_string.length ()) == first)
			//	No remainder.
			break Values;
		}
	try {range.End = Long.parseLong (range_string.substring (first, last));}
	catch (NumberFormatException exception)
		{break Values;}

	//	Length
	first = ++last;
	if (first < range_string.length ())
		{
		try {range.Length = Long.parseLong (range_string.substring (first));}
		catch (NumberFormatException exception) {}
		}
	}
if ((DEBUG & DEBUG_HEADERS) != 0)
	System.out.println ("<<< HTTP_ImageInputStream.Content_Range: " + range);
return range;
}

/**	A <i>Range</i> contains Start, End and Length values.
<p>
	The Length is not necessarily the length between Start and
	End. There are no constraints on any of the values.
*/
public class Range
{
public long
	Start	= 0,
	End		= 0,
	Length	= 0;

Range
	(
	long	start,
	long	end,
	long	length
	)
{
Start	= start;
End		= end;
Length	= length;
}

public String toString ()
{
return Long.toString (Start) + '-' + End + '/' + Length;
}
}

/*==============================================================================
	Application main
*/

public static void main
	(
	String[]    args
	)
{
if (DEBUG != DEBUG_OFF)
	System.out.println
		("*** " + ID + " ***");
String
	connection_URL = null;
boolean
	logging = false;

for (int count = 0;
	 count < args.length;
	 count++)
	{
	if (args[count].length () > 1 &&
		args[count].charAt (0) == '-')
		{
		switch (args[count].charAt (1))
			{
			case 'L':
			case 'l':
				logging = true;
				break;

			case 'H':
			case 'h':
			default:
				Usage ();
			}
		}
	else
		{
		if (connection_URL != null)
			{
			System.out.println
				("Multiple connection URLs specified -\n"
				+"    " + connection_URL + " and\n" + args[count]);
			Usage ();
			}
		connection_URL = args[count];
		}
	}

if (connection_URL == null)
	Usage ();
try
	{
	HTTP_ImageInputStream
		source = new HTTP_ImageInputStream (new URL (connection_URL));

	source.Logging (logging);
	if (DEBUG != DEBUG_OFF)
		source.Logging (true);

	System.out.println ("Check source status: "
		+ source.Check_Source ());
	}
catch (Exception exception)
	{
	System.out.println ("!!! " + exception);
	System.exit (1);
	}
}


public static void Usage ()
{
System.out.println
	(
	ID + '\n' +
	"Usage: HTTP_ImageInputStream <URL>\n"
	);
System.exit (1);
}

}
