PROGRAM create_inputfield
!
!****	create_inputfield
!
!	Purpose:
!	--------
!	This program creates an analytical field on a grid given as argument
!       with a time dimension
!       
!***	History:
!       -------
!       Version   Programmer      Date        Description
!       -------   ----------      ----        -----------
!         1.0     Sophie Valcke   2007/12/10  Creation
!*----------------------------------------------------------------
!
!** ++ calling argument
!       1- source grid acronym (CHARACTER(len=4))
!       2- analytical field (1, 2, or 3, CHARACTER(len=1))
!             1) F = 2 - cos[Pi*acos(cos(lat)cos(lon)]
!             2) F = 2 + [(cos(lat))**2]*cos(2*lon)
!             3) F = 2 + [(sin(2*lon))**16]*cos(16lon)
!
!** ++ modules and includes
!
  IMPLICIT NONE
  INCLUDE 'netcdf.inc'
!
!** ++ declarations
!
  CHARACTER(len=4)       :: cl_grd
  CHARACTER(len=1)       :: cl_fld
  CHARACTER(len=8)       :: cl_nam
  CHARACTER(len=64)      :: cl_ficgrd, cl_ficmsk
  INTEGER                :: il_fileid, il_lonid, il_latid, il_mskid 
  INTEGER                :: il_ficid, il_fldid, il_timeid
  INTEGER                :: il_ndims, il_i, il_j, il_ij, il_type
  INTEGER                :: il_nbtsteps, il_b
  INTEGER, DIMENSION(:), ALLOCATABLE :: il_dimids, il_i_dimid, il_j_dimid, il_dimidstime
  INTEGER, DIMENSION(:)  :: il_start, il_count
  INTEGER, DIMENSION(:,:), ALLOCATABLE :: il_msk
  REAL*4,  DIMENSION(:,:), ALLOCATABLE :: rl_lon
  REAL*4,  DIMENSION(:,:), ALLOCATABLE :: rl_lat
  REAL*4,  DIMENSION(:,:), ALLOCATABLE           :: rl_fld
  REAL*4,  DIMENSION(:,:,:), ALLOCATABLE         :: rl_fldtime
  REAL*4,  DIMENSION(:), ALLOCATABLE         :: rl_time
  DOUBLE PRECISION, PARAMETER                 :: two = 2.
  DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE :: dl_lon
  DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE :: dl_lat
  DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE :: dl_fld
  DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE :: dl_fldtime
  DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: dl_time
  DOUBLE PRECISION, PARAMETER    :: dp_pi=3.14159265359
  DOUBLE PRECISION, PARAMETER    :: dp_length= 1.2*dp_pi
  DOUBLE PRECISION, PARAMETER    :: dp_conv = dp_pi/180.
  LOGICAL                :: ll_dbl

!*----------------------------------------------------------------
!
! Get arguments giving source grid acronym and field type
! 
  CALL getarg(1,cl_ficgrd)
  PRINT *, 'Source grid file name = ', cl_ficgrd
  CALL getarg(2,cl_ficmsk)
  PRINT *, 'Source mask file name= ', cl_ficmsk
  CALL getarg(3,cl_grd)
  PRINT *, 'Source grid acronym = ', cl_grd
  CALL getarg(4,cl_fld)
  PRINT *, 'Analytical field number = ', cl_fld
!
! Open grids file and get grid longitudes and latitudes
!
  CALL hdlerr(NF_OPEN(cl_ficgrd, NF_NOWRITE, il_fileid))
!  
  cl_nam=cl_grd//".lon" 
  WRITE(*,*) 'cl_nam ', cl_nam
  CALL hdlerr(NF_INQ_VARID(il_fileid, cl_nam, il_lonid))
  WRITE(*,*) 'il_lonid', il_lonid
  cl_nam=cl_grd//".lat" 
  WRITE(*,*) 'cl_nam ', cl_nam
  CALL hdlerr(NF_INQ_VARID(il_fileid, cl_nam, il_latid))
  WRITE(*,*) 'il_latid', il_latid
!
  CALL hdlerr(NF_INQ_VARNDIMS(il_fileid, il_lonid, il_ndims))
  WRITE(*,*) 'il_ndims =', il_ndims
  ALLOCATE (il_dimids(il_ndims))
  CALL hdlerr(NF_INQ_VARDIMID(il_fileid, il_lonid, il_dimids))
  WRITE(*,*) 'il_dimids =', il_dimids
  CALL hdlerr(NF_INQ_VARTYPE(il_fileid, il_lonid, il_type))
  ll_dbl = .false.
  IF (il_type == NF_DOUBLE) ll_dbl = .TRUE.
  WRITE(*,*) 'il_dimids =', il_dimids
  IF (il_ndims == 1) THEN
      CALL hdlerr(NF_INQ_DIMLEN(il_fileid,il_dimids(1), il_ij))
      WRITE(*,*) 'il_ij= ', il_ij
  ELSE
      CALL hdlerr(NF_INQ_DIMLEN(il_fileid,il_dimids(1), il_i))
      WRITE(*,*) 'il_i= ', il_i
      CALL hdlerr(NF_INQ_DIMLEN(il_fileid,il_dimids(2), il_j))
      WRITE(*,*) 'il_j= ', il_j
      il_ij = il_i*il_j
      WRITE(*,*) 'il_ij= ', il_ij
  ENDIF
  IF (ll_dbl) THEN
      ALLOCATE (dl_lon(il_i, il_j))
      ALLOCATE (dl_lat(il_i, il_j))
      CALL hdlerr(NF_GET_VAR_DOUBLE (il_fileid, il_lonid, dl_lon))
      CALL hdlerr(NF_GET_VAR_DOUBLE (il_fileid, il_latid, dl_lat))
  ELSE
      ALLOCATE (rl_lon(il_i, il_j))
      ALLOCATE (rl_lat(il_i, il_j))
      CALL hdlerr(NF_GET_VAR_REAL (il_fileid, il_lonid, rl_lon))
      CALL hdlerr(NF_GET_VAR_REAL (il_fileid, il_latid, rl_lat))
  ENDIF
  CALL hdlerr(NF_CLOSE(il_fileid))
!
! Open mask file and get mask file
!
  CALL hdlerr(NF_OPEN(cl_ficmsk, NF_NOWRITE, il_fileid))
!  
  cl_nam=cl_grd//".msk" 
  write(*,*) 'cl_nam ', cl_nam
  call hdlerr(NF_INQ_VARID(il_fileid, cl_nam, il_mskid))
  write(*,*) 'il_mskid', il_mskid
!
  ALLOCATE (il_msk(il_i, il_j))
  CALL hdlerr(NF_GET_VAR_INT (il_fileid, il_mskid, il_msk))
  CALL hdlerr(NF_CLOSE(il_fileid))
!
! Create field and apply mask
!
  IF (ll_dbl) THEN
      dl_lat = dl_lat * dp_conv
      dl_lon = dl_lon * dp_conv
      ALLOCATE (dl_fld(il_i, il_j))
      IF (cl_fld == '1') THEN
          dl_fld =  two - COS(dp_pi*(ACOS(COS(dl_lat)*COS(dl_lon))/dp_length))
      ELSE IF (cl_fld == '2') THEN
          dl_fld = two + COS(dl_lat)**2*COS(two*dl_lon)
      ELSE IF (cl_fld == '3') THEN
          dl_fld = two + SIN(two*dl_lat)**16*COS(16.*dl_lon)
      ELSE
          WRITE(*,*) 'Bad analytical field number (2nd calling argument)'
          WRITE(*,*) 'should be either "1", "2" or "3"'
          STOP
      ENDIF
!
      WHERE (il_msk == 1)
          dl_fld = 0.00
      END WHERE
  ELSE
      rl_lat = rl_lat * dp_conv
      rl_lon = rl_lon * dp_conv     
      ALLOCATE (rl_fld(il_i, il_j))
      IF (cl_fld == '1') THEN
          rl_fld =  two - COS(dp_pi*(ACOS(-COS(rl_lat)*COS(rl_lon))/dp_length))
      ELSE IF (cl_fld == '2') THEN
          rl_fld = two + COS(rl_lat)**2*COS(two*rl_lon)
      ELSE IF (cl_fld == '3') THEN
          rl_fld = two + SIN(two*rl_lat)**16*COS(16.*rl_lon)
      ELSE
          WRITE(*,*) 'Bad analytical field number (2nd calling argument)'
          WRITE(*,*) 'should be either "1", "2" or "3"'
          STOP
      ENDIF
!
      WHERE (il_msk == 1)
          rl_fld = 0.00
      END WHERE
  ENDIF
!
! Create the field with the time dimension
  il_nbtsteps = 12
  IF (ll_dbl) THEN
      ALLOCATE (dl_fldtime(il_i, il_j,il_nbtsteps))
      ALLOCATE (dl_time(il_nbtsteps))
      DO il_b = 1, il_nbtsteps 
        dl_fldtime(:,:,il_b)=dl_fld(:,:)*il_b
        dl_time(il_b) = (il_b-1) * 43200 
      ENDDO
  ELSE 
      ALLOCATE (rl_fldtime(il_i, il_j,il_nbtsteps))
      ALLOCATE (rl_time(il_nbtsteps))
      DO il_b = 1, il_nbtsteps 
        rl_fldtime(:,:,il_b)=rl_fld(:,:)*il_b
        rl_time(il_b) = (il_b-1) * 43200 
      ENDDO
  ENDIF

! Create file and write the field
!
! Create file
  CALL hdlerr(NF_CREATE('SOALBEDO.nc', 0, il_ficid))
!
! Create dimensions
  ALLOCATE (il_dimidstime(3))
  CALL hdlerr(NF_DEF_DIM(il_ficid, 'il_i', il_ij,il_dimidstime(1)))
  CALL hdlerr(NF_DEF_DIM(il_ficid, 'il_j', 1,il_dimidstime(2)))
  CALL hdlerr(NF_DEF_DIM(il_ficid, 'time', NF_UNLIMITED,il_dimidstime(3)))
!
! Create variables
  CALL hdlerr(NF_DEF_VAR (il_ficid, 'SOALBEDO', il_type, 3, &
     il_dimidstime, il_fldid))
  cl_nam=cl_grd//".lon" 
  CALL hdlerr(NF_DEF_VAR (il_ficid, cl_nam, il_type, 2, il_dimidstime(1:2), il_lonid))
  WRITE(*,*) 'il_lonid = ', il_lonid
  cl_nam=cl_grd//".lat" 
  CALL hdlerr(NF_DEF_VAR (il_ficid, cl_nam, il_type, 2, il_dimidstime(1:2), il_latid))
  WRITE(*,*) 'il_latid = ', il_latid
  CALL hdlerr(NF_DEF_VAR (il_ficid, 'time', il_type, 1, &
     il_dimidstime(3), il_timeid))
!
! End of definition phase
  CALL hdlerr(NF_ENDDEF(il_ficid)) 
!
! Write the field
  ALLOCATE (il_start(3))
  ALLOCATE (il_count(3))
  il_start(:) = 1
  il_count(1) = il_ij
  il_count(2) = 1
  il_count(3) = il_nbtsteps      
!
  IF (ll_dbl) THEN
      CALL hdlerr(NF_PUT_VARA_DOUBLE (il_ficid, il_lonid, il_start(1:2), il_count(1:2), dl_lon)) 
      CALL hdlerr(NF_PUT_VARA_DOUBLE (il_ficid, il_latid, il_start(1:2), il_count(1:2), dl_lat)) 
      CALL hdlerr(NF_PUT_VARA_DOUBLE (il_ficid, il_fldid, il_start(1:3), il_count(1:3), dl_fldtime))
      CALL hdlerr(NF_PUT_VARA_DOUBLE (il_ficid, il_timeid, il_start(3), il_count(3), dl_time))
  ELSE
      CALL hdlerr(NF_PUT_VARA_REAL (il_ficid, il_lonid, il_start(1:2), il_count(1:2), rl_lon)) 
      CALL hdlerr(NF_PUT_VARA_REAL (il_ficid, il_latid, il_start(1:2), il_count(1:2), rl_lat))
      CALL hdlerr(NF_PUT_VARA_REAL (il_ficid, il_fldid, il_start(1:3), il_count(1:3), rl_fldtime))
      CALL hdlerr(NF_PUT_VARA_DOUBLE (il_ficid, il_timeid, il_start(3), il_count(3), rl_time))
  ENDIF
!
! Close the file
  CALL hdlerr(NF_CLOSE(il_ficid))
!
END PROGRAM create_inputfield
!
!*----------------------------------------------------------------
!*----------------------------------------------------------------
!
SUBROUTINE hdlerr(istatus)

  INTEGER                 :: istatus
  INCLUDE 'netcdf.inc'

  IF (istatus .ne. NF_NOERR) THEN
      print *, NF_STRERROR(istatus)
      stop 'stopped'
  ENDIF

  RETURN

END SUBROUTINE hdlerr
!
!*----------------------------------------------------------------
!*----------------------------------------------------------------
!
