/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


/**
   \file cdw_dialog.c

   File with implementation of simple dialog window of two types:
   dialog window with one or more buttons, and dialog window with
   one-line text input. Both types of dialog window display
   a message/prompt.
*/


#include <string.h>
#include <stdlib.h>

#include "cdw_dialog.h"
#include "cdw_button.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_window.h"
#include "cdw_string.h"
#include "cdw_form.h"
#include "cdw_ncurses.h"


/* a dialog can be either dialog with input field, or dialog with
   buttons; in future it may be possible that the two will be
   possible at the same time */
enum {
	CDW_DIALOG_BUTTONS = 1,
	CDW_DIALOG_INPUT = 2   /* Unused at the moment. */
};


/* 4 buttons should be enough for everyone */
#define CDW_DIALOG_N_BUTTONS 4

/* basic building block for creating more advanced dialogs */
typedef struct {
	/* type of dialog window: CDW_DIALOG_BUTTONS or
	   CDW_DIALOG_INPUT, but always with text on top */
	int dialog_type;
	cdw_colors_t colors;

	WINDOW *window;
	int begin_y;
	int begin_x;

	/* space between dialog window borders and content of the window:
	   the text area and buttons/input field; pad_x may be 0 if
	   more space is needed to display text without truncations */
	int pad_y;
	int pad_x;

	/* size of content (txt area + (buttons or input field)) of dialog
	   window, sum this with pad_x and pad_y to get total dialog window size */
        int n_rows_internal;
	int n_cols_internal;

	/* text area in upper part of dialog; it can be one-line prompt, but
	   it can be multi-line text too */
	struct {
		WINDOW *subwindow;
		int n_rows;
		int n_cols;
	} txt;

	/* area where buttons or input field is displayed */
	struct {
		int n_cols;
		int n_rows; /* will be always (?) equal one, see comment below for explanation */

		/* buttons dialog:
		   all buttons are in one row, the row's number is begin_y;

		   input line dialog:
		   input line occupies one row in dialog window, the row's number is begin_y */
		int begin_y;

		/* empty line between text area and entry (input) area below */
		int spacer_y;
	} input;

	/* if dialog->type is CDW_DIALOG_BUTTONS, then this struct
	   stores buttons visible below text area (in "input" area) */
	struct {
		CDW_BUTTON *button;
		char *label;
		int label_len;
		bool used;
	} button[CDW_DIALOG_N_BUTTONS];
	/* currently selected button, used e.g. in driver */
	int button_current;
	/* how many buttons and of which kind */
	int button_types;

} CDW_DIALOG_BASE;





/* symbolic names to control buttons in cdw_buttons_dialog()
   (e.g. to index buttons table) */
enum {
	BUTTON_OK     = 0,
	BUTTON_YES    = 1,
	BUTTON_NO     = 2,
	BUTTON_CANCEL = 3
};



static CDW_DIALOG_BASE *cdw_dialog_base_new(int dialog_type);
static void cdw_dialog_base_delete(CDW_DIALOG_BASE **dialog);
static cdw_rv_t cdw_dialog_base_sizes(CDW_DIALOG_BASE *dialog, const char *message);
static cdw_rv_t cdw_dialog_base_input_sizes(CDW_DIALOG_BASE *dialog);
static int      cdw_dialog_base_input_sizes_max(CDW_DIALOG_BASE *dialog);
static cdw_rv_t cdw_dialog_base_txt_sizes(CDW_DIALOG_BASE *dialog, const char *message);
static int      cdw_dialog_base_txt_sizes_rows(CDW_DIALOG_BASE *dialog, const char *message);
static void cdw_dialog_base_display_basics(CDW_DIALOG_BASE *dialog, const char *title, const char *message, int align);


static void cdw_dialog_create_button_labels(CDW_DIALOG_BASE *dialog);
static cdw_rv_t cdw_buttons_dialog_driver(CDW_DIALOG_BASE *dialog);
static void cdw_dialog_display_buttons(CDW_DIALOG_BASE *dialog);




/**
   \brief Create basis of new dialog widget

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   \param dialog_type - CDW_DIALOG_BUTTONS or CDW_DIALOG_INPUT

   \return pointer to new widget on success
   \return NULL on error
*/
CDW_DIALOG_BASE *cdw_dialog_base_new(int dialog_type)
{
	CDW_DIALOG_BASE *base = (CDW_DIALOG_BASE *) malloc(sizeof (CDW_DIALOG_BASE));
	if (!base) {
		cdw_vdm ("ERROR: failed to allocate memory for dialog base\n");
		return (CDW_DIALOG_BASE *) NULL;
	}

	for (int i = 0; i < CDW_DIALOG_N_BUTTONS; i++) {
		base->button[i].button = (CDW_BUTTON *) NULL;
		base->button[i].label = (char *) NULL;
		base->button[i].used = false;
		base->button[i].label_len = 0;
	}

	/* These symbolic constants are used as loop indexes
	   and boundaries, make sure that they weren't modified */
	cdw_assert (BUTTON_OK == 0 && BUTTON_YES == 1 && BUTTON_NO == 2 && BUTTON_CANCEL == 3,
		    "ERROR: values of button names have been modified");

	/* currently selected button, set by cdw_dialogbox_display_buttons(),
	   based on dialog_type */
	base->button_current = BUTTON_OK;

	base->window = (WINDOW *) NULL;
	base->begin_y = 0;
	base->begin_x = 0;
	/* keep some space between dialog window borders
	   and everything inside the window */
	base->pad_y = 1;
	base->pad_x = 1;

	/* keep space between text window and input area below the window */
	base->input.spacer_y = 1;
	base->input.n_cols = 0;

	base->txt.subwindow = (WINDOW *) NULL;
	/* initial text area size */
	base->txt.n_rows = 0;
	/* 40 is half of minimal supported terminal width, seems ok */
	base->txt.n_cols = 40;

	base->colors = CDW_COLORS_DIALOG;

	base->dialog_type = dialog_type;
	base->button_types = 0;

	return base;
}





/**
   \brief Deallocate resources allocated by cdw_dialog_base_new()

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Wrapper for few calls to free().
   Also frees variable pointed to by argument and sets it to NULL.

   \param dialog - pointer to widget created by cdw_dialog_base_new()
*/
void cdw_dialog_base_delete(CDW_DIALOG_BASE **dialog)
{
	for (int i = 0; i < CDW_DIALOG_N_BUTTONS; i++) {
		if ((*dialog)->button[i].button) {
			cdw_assert ((*dialog)->dialog_type == CDW_DIALOG_BUTTONS,
				    "ERROR: button #%d is not null, but dialog type != CDW_DIALOG_BUTTONS\n", i);
			cdw_button_delete(&((*dialog)->button[i].button));
			cdw_assert (!(*dialog)->button[i].button,
				    "ERROR: delete didn't set button pointer to NULL\n");
		}
	}

	if ((*dialog)->txt.subwindow) {
		delwin((*dialog)->txt.subwindow);
		(*dialog)->txt.subwindow = (WINDOW *) NULL;
	}

	if ((*dialog)->window) {
		delwin((*dialog)->window);
		(*dialog)->window = (WINDOW *) NULL;
	}

	free(*dialog);
	*dialog = (CDW_DIALOG_BASE *) NULL;

	return;
}





/**
   \brief Calculate size of dialog's "input" area

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Function calculates size of area in which buttons or input
   widget will be put. The exact size is not known at the beginning,
   so it has to be calculated using this function.

   \param dialog - dialog window for which to calculate input area

   \return CDW_OK on success
   \return CDW_ERROR on failure
*/
cdw_rv_t cdw_dialog_base_input_sizes(CDW_DIALOG_BASE *dialog)
{
	/* FIXME: magic number, arbitrary */
	const int nice_input_width = 50;

	/* maximal width of input field or buttons area, -2 is for window borders */
	const int max_cols = COLS - 2 - 2 * dialog->pad_x;

	if (dialog->dialog_type == CDW_DIALOG_BUTTONS) {
		for (int i = BUTTON_OK; i <= BUTTON_CANCEL; i++) {
			if (dialog->button[i].used) {
				dialog->input.n_cols += dialog->button[i].label_len;
			}
		}

	} else if (dialog->dialog_type == CDW_DIALOG_INPUT) {
		/* input from user may have any length, but in order to not
		   to have input field as wide as main app window, or very
		   narrow in case of small limit, let's set it's width to
		   sth that will make the input field look "nicely" */
		dialog->input.n_cols = nice_input_width;
	} else {
		cdw_assert (0, "ERROR: unknown dialog type %d\n", dialog->dialog_type);
	}

	cdw_sdm ("INFO: initial required width of dialog base is %zd\n", dialog->input.n_cols);

	/* we have some expected, ideal number of columns to display given
	   set of buttons, or to display input field that would fit whole
	   input; now let's deal with limitations imposed by current screen */

	dialog->input.n_rows = 1; /* buttons have to fit in one row, input field always takes one row */

	bool buttons_crammed = false;
	if (dialog->input.n_cols > max_cols) {
		dialog->input.n_cols = max_cols;

		if (dialog->dialog_type == CDW_DIALOG_BUTTONS) {
			buttons_crammed = true;
			/* in case of buttons dialog the button labels may
			   overlap */
			cdw_vdm ("ERROR: buttons in a dialog will overlap\n");
		} else {
			/* in case of input dialog the input field
			   won't display whole content at once,
			   nothing wrong with that */
		}
	}

	cdw_sdm ("INFO: final required width of dialog input area is %zd\n", dialog->input.n_cols);

	if (buttons_crammed) {
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Calculate all internal sizes of dialog box

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Function calls cdw_dialog_base_input_sizes()
   and cdw_dialog_base_txt_sizes(), and
   does some other calculations in order to get x/y size of
   dialog window that will be presented to user.

   Function returns CDW_ERROR on failure, but it may still be possible
   to display a dialog, but its content will be displayed at least
   partially incorrectly.

   \param dialog - dialog base for which the size will be calculated
   \param message - message that will be displayed in dialog window

   \return CDW_OK on success
   \return CDW_ERROR on failure
*/
cdw_rv_t cdw_dialog_base_sizes(CDW_DIALOG_BASE *dialog, const char *message)
{
	cdw_rv_t crv1 = cdw_dialog_base_input_sizes(dialog);
	if (crv1 != CDW_OK) {
		cdw_vdm ("ERROR: failed to calculate satisfactory size of input area, buttons may be crammed\n");
	}

	cdw_rv_t crv2 = cdw_dialog_base_txt_sizes(dialog, message);
	if (crv2 != CDW_OK) {
		cdw_vdm ("ERROR: failed to calculate satisfactory sizes of txt subwindow, text may be clipped\n");
	}

	/* at this moment functions calculating sizes for input area and
	   textarea have specified their size requirements and have set
	   following fields in dialog data structure:

	   dialog->n_rows_input
	   dialog->n_cols_input
	   dialog->n_rows_txt
	   dialog->n_cols_txt

	   now we must resolve n_rows_internal and n_cols_internal */

	if (dialog->input.n_cols > dialog->txt.n_cols) {
		dialog->n_cols_internal = dialog->input.n_cols;
	} else {
		dialog->n_cols_internal = dialog->txt.n_cols;
	}

	int max = cdw_dialog_base_input_sizes_max(dialog);
	if (max < dialog->txt.n_rows) {

		dialog->pad_y = 0;
		max = cdw_dialog_base_input_sizes_max(dialog);

		if (max < dialog->txt.n_rows) {

			dialog->input.spacer_y = 0;
			max = cdw_dialog_base_input_sizes_max(dialog);

			if (max < dialog->txt.n_rows) {
				dialog->txt.n_rows = max;
			}
		}
	}

	dialog->n_rows_internal = dialog->txt.n_rows + dialog->input.spacer_y + dialog->input.n_rows;
	dialog->input.begin_y = 1 + dialog->pad_y + dialog->txt.n_rows + dialog->input.spacer_y;

	if (crv1 != CDW_OK || crv2 != CDW_OK) {
		cdw_vdm ("ERROR: something went wrong when calculating sizes in dialogbox, expect layout problems\n");
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Wrapper for a common simple calculation

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   \param dialog - dialog widget for which a function calculates x/y sizes

   \return number of lines that can be used for a specific part of dialog window
*/
int cdw_dialog_base_input_sizes_max(CDW_DIALOG_BASE *dialog)
{
	return LINES - 2                  /* -2 for window borders */
		- 2 * dialog->pad_y       /* may be already set to 0 by calculate_subwindow_txt_size() */
		- dialog->input.n_rows    /* this is not to be changed anymore */
		- dialog->input.spacer_y; /* additional row above and below input area */
}





/**
   \brief Calculate sizes of text area of a dialog

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Calculate x/y sizes of text area in dialog widget. Adjust
   spacing of dialog's layout to fit given \p message if necessary.

   If the message is too long (or the dialog window too small), the function
   may be unable to recalculate size of text area and to adjust layout of
   elements in the dialog. The result may be truncation of text in dialog window.
   The text will be visible, but only partially. Function returns CDW_ERROR
   in such situation.

   \param dialog - dialog widget for which a function calculates x/y sizes
   \param message - message that will be displayed in the text area of a widget

   \return CDW_OK on success (i.e. whole text will fit into dialog window)
   \return CDW_ERROR on problems (i.e. some text may be invisible in dialog window)
*/
cdw_rv_t cdw_dialog_base_txt_sizes(CDW_DIALOG_BASE *dialog, const char *message)
{
	/* FIXME: magic number, arbitrary value */
	dialog->txt.n_cols = 40;
	int n_rows = cdw_dialog_base_txt_sizes_rows(dialog, message);

	if (n_rows > LINES) {
		/* message won't fit into dialog window of current width,
		   stretch the width .. */
		/* first -2 is for window borders */
		dialog->txt.n_cols = COLS - 2 - 2 * dialog->pad_x;
		n_rows = cdw_dialog_base_txt_sizes_rows(dialog, message);

		if (n_rows > LINES) {
			/* get rid of non-critical space... */
			dialog->pad_x = 0;
			dialog->txt.n_cols = COLS - 2 - 2 * dialog->pad_x;
			n_rows = cdw_dialog_base_txt_sizes_rows(dialog, message);

			if (n_rows > LINES) {
				/* last try, get rid of more non-critical space */
				dialog->pad_y = 0;
				n_rows = cdw_dialog_base_txt_sizes_rows(dialog, message);
			}
		}

	}

	if (n_rows > LINES) {
		cdw_vdm ("ERROR: failed to calculate satisfactory sizes of text area, text may be clipped\n");
		cdw_vdm ("ERROR: n_rows = %d, LINES = %d\n", n_rows, LINES);
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Wrapper for a common simple calculation

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Function calculates one specific value: number of rows of text area in
   a dialog window.
   The value returned by the function includes spaces above and below
   text area.

   \param dialog - dialog widget for which to perform calculations
   \param message - message that will be displayed in text area of the dialog

   \return number of lines that will be used by text area
*/
int cdw_dialog_base_txt_sizes_rows(CDW_DIALOG_BASE *dialog, const char *message)
{
	dialog->txt.n_rows = (int) cdw_string_count_lines(message, (size_t) dialog->txt.n_cols);
	int n_rows = dialog->txt.n_rows + 2 * dialog->pad_y + dialog->input.spacer_y + dialog->input.n_rows;

	return n_rows;
}





/**
   \brief Create and display minimum content of dialog widget

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Create and display dialog's window, window's border, and message/prompt
   in the window. Use \p dialog as a basis of the widget.
   Don't display input line or buttons, leave it to other functions.

   \param dialog - dialog widget to display
   \param title - title of the dialog window
   \param message - prompt/message to be displayed in the window
   \param align - align of message text
*/
void cdw_dialog_base_display_basics(CDW_DIALOG_BASE *dialog, const char *title, const char *message, int align)
{
	/* +2 is for borders */
	int n_rows = dialog->n_rows_internal + 2 * dialog->pad_y + 2;
	int n_cols = dialog->n_cols_internal + 2 * dialog->pad_x + 2;
	dialog->begin_y = (LINES - n_rows) / 2;
	dialog->begin_x = (COLS - n_cols) / 2;

	dialog->window = cdw_window_new((WINDOW *) NULL,
					n_rows, n_cols, dialog->begin_y, dialog->begin_x,
					dialog->colors,
					title, (char *) NULL);

	dialog->txt.subwindow = cdw_window_new(dialog->window,
					       dialog->txt.n_rows, dialog->txt.n_cols,
					       1 + dialog->pad_y, 1 + dialog->pad_x,
					       dialog->colors,
					       (char *) NULL, (char *) NULL);

	cdw_window_print_message(dialog->txt.subwindow, message, align);
	wrefresh(dialog->txt.subwindow);

	return;
}





/**
   \brief Initialize labels of buttons

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Initialize labels of all buttons. The labels are hardwired,
   there is no way for code using this widget to provide custom labels.

   \param dialog - dialog base for which to initialize labels
*/
void cdw_dialog_create_button_labels(CDW_DIALOG_BASE *dialog)
{
	/* 2TRANS: button label, please keep as short as possible */
	dialog->button[BUTTON_OK].label = _("OK");
	/* 2TRANS: button label, please keep as short as possible */
	dialog->button[BUTTON_YES].label = _("Yes");
	/* 2TRANS: button label, please keep as short as possible */
	dialog->button[BUTTON_NO].label = _("No");
	/* 2TRANS: button label, please keep as short as possible */
	dialog->button[BUTTON_CANCEL].label = _("Cancel");

	dialog->button[BUTTON_OK].label_len =     (int) strlen(dialog->button[BUTTON_OK].label);
	dialog->button[BUTTON_YES].label_len =    (int) strlen(dialog->button[BUTTON_YES].label);
	dialog->button[BUTTON_NO].label_len =     (int) strlen(dialog->button[BUTTON_NO].label);
	dialog->button[BUTTON_CANCEL].label_len = (int) strlen(dialog->button[BUTTON_CANCEL].label);

	return;
}





/**
   \brief Driver function reacting to keys pressed in buttons dialog window

   \date Function's top-level comment reviewed on 2012-01-07
   \date Function's body reviewed on 2012-01-07

   Function reads keys pressed by user when a dialog box with buttons
   is displayed, and reacts appropriately: moving focus from one key
   to another, and returning appropriate value when user hits ENTER,
   ESCAPE, or other specific key.

   \param dialog - dialog to be controlled

   \return CDW_OK if user selected 'OK' or 'Yes' button
   \return CDW_CANCEL if user selected 'Cancel' button or pressed ESCAPE/'q'/'Q' key
   \return CDW_NO if user selected 'No' button
 */
cdw_rv_t cdw_buttons_dialog_driver(CDW_DIALOG_BASE *dialog)
{
	cdw_assert (dialog, "ERROR: cannot operate on NULL dialog\n");
	cdw_assert (dialog->window, "ERROR: cannot operate on NULL window\n");

	int key = 0;
	/* handle cursor keys - highlight selected button */
	while ((key = wgetch(dialog->window)) != ERR) {
		if (key == 'q' || key == 'Q') {
			key = CDW_KEY_ESCAPE;
			break;
		}
		if (key == CDW_KEY_ESCAPE || key == CDW_KEY_ENTER) {
			break;
		}

		cdw_button_unfocus(dialog->button[dialog->button_current].button);

		/* There are four possible buttons in a dialog window.
		   Some of them may be passive/invisible/inactive.
		   Search for active one, going left or right. */

		/* 10 is probably an overkill; ~4 will be enough */
		for (int i = 0; i < 10; i++) {
			if (key == KEY_LEFT) {
				dialog->button_current--;
				if (dialog->button_current < BUTTON_OK) {
					dialog->button_current = BUTTON_CANCEL;
				}
			} else if (key == KEY_RIGHT || key == CDW_KEY_TAB) {
				dialog->button_current++;
				if (dialog->button_current > BUTTON_CANCEL) {
					dialog->button_current = BUTTON_OK;
				}

			} else {
				;
			}
			if (dialog->button[dialog->button_current].used) {
				/* found an existing/initialized/visible
				   button; stop searching and focus on it */
				break;
			}
		}
		cdw_button_focus(dialog->button[dialog->button_current].button);
	}

	if (key == CDW_KEY_ENTER) {
		if (dialog->button_current == BUTTON_OK || dialog->button_current == BUTTON_YES) {
			return CDW_OK;
		} else if (dialog->button_current == BUTTON_NO) {
			return CDW_NO;
		} else {
			return CDW_CANCEL;
		}
	} else { /* CDW_KEY_ESCAPE */
		return CDW_CANCEL;
	}
}





/**
   \brief Build buttons and display them in proper places of dialog window

   \date Function's top-level comment reviewed on 2012-01-08
   \date Function's body reviewed on 2012-01-08

   The function initializes button labels and buttons, calculates position
   of buttons and displays puts them in proper order and position - all
   that depending on type of dialog.

   \p dialog_type can be one of those: DIALOG_OK, DIALOG_OK_CANCEL,
   DIALOG_YES_NO_CANCEL

   \p colors is usually symbolic value passed to function by parent widget.

   \p current_button will be initialized by the function

   \param window - window in which you want to put buttons (dialog window)
   \param current_button - button that will have initial focus
   \param dialog_type - type of dialog in which caller wants to draw buttons
   \param colors - color scheme of the button
*/
void cdw_dialog_display_buttons(CDW_DIALOG_BASE *dialog)
{
	cdw_assert (dialog->window, "ERROR: cannot display buttons in NULL window\n");

	/* Placing buttons is a bit tricky if you consider that
	   translations may have widths different than original
	   strings.
	   Width of whole dialog window is divided into one, two
	   or three equal slots, depending on number of buttons.
	   Then position of start of button in each slot is
	   calculated. */

	int n_cols = getmaxx(dialog->window) - 2;

	/* how many columns at most can be occupied by a button and
	   a space around the button; for N buttons: n_cols / N */
	int button_cols = 0;

	/* draw initial dialog box */
	if (dialog->button_types == CDW_BUTTONS_OK) {
		dialog->button_current = BUTTON_OK;
		dialog->button[BUTTON_OK].used = true;
		button_cols = n_cols/1;

	} else if (dialog->button_types == CDW_BUTTONS_OK_CANCEL) {
		dialog->button_current = BUTTON_OK;
		dialog->button[BUTTON_OK].used = true;
		dialog->button[BUTTON_CANCEL].used = true;
		button_cols = n_cols/2;

	} else if (dialog->button_types == CDW_BUTTONS_YES_NO_CANCEL) {
		dialog->button_current = BUTTON_YES;
		dialog->button[BUTTON_YES].used = true;
		dialog->button[BUTTON_NO].used = true;
		dialog->button[BUTTON_CANCEL].used = true;
		button_cols = n_cols/3;

	} else {
		;
	}

	int a = 0;
	for (int i = BUTTON_OK; i <= BUTTON_CANCEL; i++) {
		/* check if i-th button should be created */
		if (dialog->button[i].used) {
			int col = (a++ * button_cols)
				+ ((button_cols - (dialog->button[i].label_len + 2)) / 2);

			dialog->button[i].button =
				cdw_button_new(dialog->window,
					       dialog->input.begin_y, col,
					       dialog->button[i].label,
					       dialog->colors);
		}
	}

	cdw_button_focus(dialog->button[dialog->button_current].button);

	return;
}





/**
   \brief Show dialog window with title, message and buttons

   \date Function's top-level comment reviewed on 2012-01-08
   \date Function's body reviewed on 2012-01-08

   Show dialog window with \p title and \p message. Depending on type of
   dialog (\p button_types) user is presented with one, two or three
   buttons.

   Function returns value of selected button. If user pressed Enter then it is:
   CDW_OK for 'Yes' or 'OK', CDW_CANCEL for 'Cancel', CDW_NO for 'No'.

   If user pressed Escape then function returns CDW_CANCEL.

   Background color depends on \p colors value.

   \param title - dialog window title
   \param message - message displayed in dialog window
   \param button_types - number and type of buttons, one of CDW_BUTTONS_* values
   \param colors - color scheme of dialog

   \return value of type cdw_rv_t, corresponding to selected button
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_buttons_dialog(const char *title, const char *message, int button_types, cdw_colors_t colors)
{
	cdw_assert (button_types == CDW_BUTTONS_OK
		    || button_types == CDW_BUTTONS_OK_CANCEL
		    || button_types == CDW_BUTTONS_YES_NO_CANCEL,

		    "ERROR: using wrong dialog button types %d\n", button_types);

	CDW_DIALOG_BASE *dialog = cdw_dialog_base_new(CDW_DIALOG_BUTTONS);
	if (!dialog) {
		cdw_vdm ("ERROR: failed to get new dialog base\n");
		return CDW_ERROR;
	}

	dialog->colors = colors;
	dialog->button_types = button_types;

	cdw_dialog_create_button_labels(dialog);

	/* debug code */
	/* char *message = "\n  a  \n  b  \n  c  \n  d  \n  e  \n  f  \n  g  \n  h  \n  i  \n  j  \n  k  \n  l  \n  m  \n  n  \n  o  \n  p  \n  q  \n  r  \n  s  \n  t  \n  u  \n  w"; */
	/* char *message = "This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.\n\nThis program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.\n\nYou should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA."; */

	cdw_dialog_base_sizes(dialog, message);
	cdw_dialog_base_display_basics(dialog, title, message, CDW_ALIGN_CENTER);

	cdw_dialog_display_buttons(dialog);

	redrawwin(dialog->window);
	wrefresh(dialog->window);

	cdw_rv_t retval = cdw_buttons_dialog_driver(dialog);

	cdw_dialog_base_delete(&dialog);

	return retval;
}






/* ***************** */


#define CDW_INPUT_DIALOG_N_FIELDS 5

typedef struct {
	WINDOW *window;
	WINDOW *subwindow;

	/* This pointer will point to string owned by client code. */
	const char *prompt_message;

	cdw_colors_t colors;

	cdw_form_t *cdw_form;
	FIELD *fields[CDW_INPUT_DIALOG_N_FIELDS + 1];

	char *buffer; /* Buffer is owned by caller. */
	int chars_max;

	int begin_x;
	int begin_y;

	int n_cols;
	int n_rows;

	int sub_begin_x;
	int sub_begin_y;

	int sub_n_cols;
	int sub_n_rows;

	int txt_n_rows; /* Height of prompt message. */

} CDW_INPUT_DIALOG;


static CDW_INPUT_DIALOG *input_dialog = (CDW_INPUT_DIALOG *) NULL;


static cdw_rv_t cdw_input_dialog_init(CDW_INPUT_DIALOG **dialog, const char *message, int chars_max, char *buffer);
static void     cdw_input_dialog_delete(CDW_INPUT_DIALOG **d);
static cdw_rv_t cdw_input_dialog_build(const char *title, const char *message, int chars_max, char *buffer);
static cdw_rv_t cdw_input_dialog_sizes(CDW_INPUT_DIALOG *dialog);
static int      cdw_input_dialog_driver(CDW_INPUT_DIALOG *dialog);
static char *   cdw_input_dialog_prompt_message_maker(int n_cols);





/* Some layout constraints. */
static const int first_col =  0;


enum {
	f_prompt_t,
	f_input_l,
	f_input_i,
	f_ok_b,
	f_cancel_b
};





cdw_rv_t cdw_input_dialog(const char *title, const char *message, int chars_max, char *buffer)
{
	cdw_rv_t crv = cdw_input_dialog_build(title, message, chars_max, buffer);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to init wizard\n");
		cdw_input_dialog_delete(&input_dialog);
		return CDW_ERROR;
	}

	int rv = cdw_input_dialog_driver(input_dialog);
	cdw_input_dialog_delete(&input_dialog);

	if (rv == CDW_KEY_ENTER) {
		return CDW_OK;
	} else {
		return CDW_CANCEL;
	}
}





cdw_rv_t cdw_input_dialog_init(CDW_INPUT_DIALOG **d, const char *message, int chars_max, char *buffer)
{
	*d = (CDW_INPUT_DIALOG *) malloc(sizeof (CDW_INPUT_DIALOG));
	if (!*d) {
		cdw_vdm ("ERROR: failed to allocate memory for input line dialog\n");
		return CDW_ERROR;
	}

	(*d)->window = (WINDOW *) NULL;
	(*d)->subwindow = (WINDOW *) NULL;

	(*d)->prompt_message = message;

	(*d)->colors = CDW_COLORS_DIALOG;

	(*d)->cdw_form = (cdw_form_t *) NULL;

	for (int i = 0; i < CDW_INPUT_DIALOG_N_FIELDS + 1; i++) {
		(*d)->fields[i] = (FIELD *) NULL;
	}

	(*d)->buffer = buffer;
	(*d)->chars_max = chars_max;

	(*d)->begin_x = 0;
	(*d)->begin_y = 0;

	(*d)->n_cols = 0;
	(*d)->n_rows = 0;

	(*d)->sub_begin_x = 0;
	(*d)->sub_begin_y = 0;

	(*d)->sub_n_cols = 0;
	(*d)->sub_n_rows = 0;

	(*d)->txt_n_rows = 0; /* Height of prompt message. */

	return CDW_OK;
}





void cdw_input_dialog_delete(CDW_INPUT_DIALOG **d)
{
	cdw_assert (d, "ERROR: passed NULL pointer to the function\n");
	if (!*d) {
		cdw_vdm ("WARNING: passed NULL input dialog to the function\n");
		return;
	}

	/* This order of these four function calls minimizes number of
	   problems reported by valgrind. */
	cdw_form_delete_form_objects((*d)->cdw_form);
	cdw_window_delete(&(*d)->subwindow);
	cdw_window_delete(&(*d)->window);
	cdw_form_delete(&((*d)->cdw_form));

	free(*d);
	*d = (CDW_INPUT_DIALOG *) NULL;

	return;
}





int cdw_input_dialog_driver(CDW_INPUT_DIALOG *d)
{
	int fi = f_input_i; /* Initial focus on input. */
	int key = 'a';      /* Safe initial value. */

	while (key != CDW_KEY_ESCAPE && key != 'q' && key != 'Q') {
		key = cdw_form_driver(d->cdw_form, fi);
		fi = field_index(current_field(d->cdw_form->form));

		if (key == CDW_KEY_ENTER) {
			cdw_assert (fi == f_ok_b, "ERROR: driver returns KEY_ENTER for widget other than OK button (%d)\n", fi);

			/* flush */
			form_driver(d->cdw_form->form, REQ_VALIDATION);

			/* Thanks to using safe input line, if we got
			   this far, the string in buffer is safe. */
			char *b = field_buffer(d->cdw_form->fields[f_input_i], 0);
			b = cdw_string_rtrim(b);
			strncpy(d->buffer, b, (size_t) d->chars_max);
			d->buffer[d->chars_max] = '\0';

			return CDW_KEY_ENTER;
		} else if (key == CDW_KEY_ESCAPE) {
			return CDW_KEY_ESCAPE;
		} else {
			;
		}
	}

	return CDW_KEY_ESCAPE;
}





static cdw_form_text_maker_t text_makers[] = {
	cdw_input_dialog_prompt_message_maker
};





char *cdw_input_dialog_prompt_message_maker(int n_cols)
{
	/* 2TRANS: this is a description in one of
	   configuration window panels; a series of
	   dropdown widgets is displayed below the label */
	return cdw_string_wrap(input_dialog->prompt_message, (size_t) n_cols, CDW_ALIGN_LEFT);
}





cdw_rv_t cdw_input_dialog_build(const char *title, const char *message, int chars_max, char *buffer)
{
	cdw_rv_t crv = cdw_input_dialog_init(&input_dialog, message, chars_max, buffer);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to get new input line dialog\n");
		return CDW_ERROR;
	}

	CDW_INPUT_DIALOG *d = input_dialog;

	d->cdw_form = cdw_form_new(CDW_INPUT_DIALOG_N_FIELDS);
	if (!d->cdw_form) {
		cdw_vdm ("ERROR: failed to create cdw form\n");
		return CDW_ERROR;
	}

	d->cdw_form->fields = d->fields;

	cdw_input_dialog_sizes(d);

	d->cdw_form->window = cdw_window_new((WINDOW *) NULL,
					     d->n_rows, d->n_cols,
					     d->begin_y, d->begin_x,
					     CDW_COLORS_DIALOG,
					     title,
					     /* 2TRANS: this is tip at the bottom
						of window - user can switch between
						window elements using tab key */
					     _("Use 'Tab' key to move"));

	if (!d->cdw_form->window) {
		cdw_vdm ("ERROR: failed to create window\n");
		return CDW_ERROR;
	}

	d->cdw_form->subwindow = cdw_window_new(d->cdw_form->window,
						d->sub_n_rows, d->sub_n_cols,
						d->sub_begin_y, d->sub_begin_x,
						CDW_COLORS_DIALOG,
						(char *) NULL, (char *) NULL);

	if (!d->cdw_form->subwindow) {
		cdw_vdm ("ERROR: failed to create subwindow\n");
		return CDW_ERROR;
	}

	int input_cols = 0;
	if (d->chars_max > d->sub_n_cols) {
		input_cols = d->sub_n_cols;
	} else {
		input_cols = d->chars_max;
	}

	int last_row = d->sub_n_rows - 2;

	cdw_form_descr_t descr[] = {
		/* type                            begin_y         begin_x      n_cols           n_lines         field enum    data1                        data2 */
		{ CDW_WIDGET_ID_TEXT,              0,              first_col,   d->sub_n_cols,   d->txt_n_rows,  f_prompt_t,   text_makers,                     0 },

		{ CDW_WIDGET_ID_DYNAMIC_LABEL,     last_row - 3,   first_col,   d->sub_n_cols,   1,              f_input_l,    "",                              0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   last_row - 2,   first_col,   input_cols,      1,              f_input_i,    d->buffer,   (size_t) d->chars_max },

		/* 2TRANS: button label */
		{ CDW_WIDGET_ID_BUTTON,            last_row,       3,           1,               1,              f_ok_b,       _("OK"),         CDW_COLORS_DIALOG },

		/* 2TRANS: button label */
		{ CDW_WIDGET_ID_BUTTON,            last_row,       23,          1,               1,              f_cancel_b,   _("Cancel"),     CDW_COLORS_DIALOG },

		/* guard */
		{ -1,                              0,              0,           0,               0,              0,            (void *) NULL,                   0 }};


	d->cdw_form->n_fields = CDW_INPUT_DIALOG_N_FIELDS;

	crv = cdw_form_description_to_fields(descr, d->cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	}

	d->cdw_form->form = cdw_ncurses_new_form(d->cdw_form->window,
						 d->cdw_form->subwindow,
						 d->cdw_form->fields);
	if (!d->cdw_form->form) {
		cdw_vdm ("ERROR: failed to create form\n");
		return CDW_ERROR;
	}


	cdw_form_add_return_chars(d->cdw_form, CDW_KEY_ENTER, CDW_KEY_ESCAPE, 'q', 'Q', 0);

	cdw_form_set_button_return_key(d->cdw_form, f_ok_b, CDW_KEY_ENTER);
	cdw_form_set_button_return_key(d->cdw_form, f_cancel_b, CDW_KEY_ESCAPE);

	cdw_form_bind_input_and_label(d->cdw_form, f_input_i, f_input_l);


	/* Make sure that cdw widgets are presented properly. */
	cdw_form_redraw_widgets(d->cdw_form);


	wrefresh(d->cdw_form->subwindow);
	wrefresh(d->cdw_form->window);

	return CDW_OK;

}





static const int cdw_dialog_sub_rows_diff = 1;
static const int cdw_dialog_sub_cols_diff = 1;


cdw_rv_t cdw_input_dialog_sizes(CDW_INPUT_DIALOG *d)
{
	int cols_space = 2 * cdw_dialog_sub_cols_diff; /* Horizontal space between parent window borders and content of window. */
	int rows_space = 2 * cdw_dialog_sub_rows_diff; /* Vertical space between parent window borders and content of window. */

	d->n_cols = 50;
	d->sub_n_cols = d->n_cols - 2 * cols_space;

	d->txt_n_rows = (int) cdw_string_count_lines(d->prompt_message, (size_t) d->sub_n_cols);

	d->n_rows = 2 * rows_space        /* Parent window should be larger than subwindow by X rows on top and on bottom. */
		+ d->txt_n_rows           /* Prompt on top. */
		+ 1                       /* Space between prompt and rest of dialog. */
		+ 1                       /* Dynamic label for "unsafe char" message. */
		+ 1                       /* Safe input line. */
		+ 1                       /* Space between safe input line and buttons. */
		+ 1                       /* Buttons row. */
		+ 1;                      /* Space between buttons row and bottom border of parent window. */

	d->sub_n_rows = d->n_rows - 2 * rows_space;

	d->begin_y = ((LINES - d->n_rows) / 2) - 2;
	d->begin_x = (COLS - d->n_cols) / 2;

	d->sub_begin_y = 2 * cdw_dialog_sub_rows_diff;
	d->sub_begin_x = 2 * cdw_dialog_sub_cols_diff;

	cdw_vdm ("INFO: input dialog base sizes:\n");
	cdw_vdm ("INFO:     n_cols = %d\n", d->n_cols);
	cdw_vdm ("INFO:     n_rows = %d\n", d->n_rows);
	cdw_vdm ("INFO: sub n_cols = %d\n", d->sub_n_cols);
	cdw_vdm ("INFO: sub n_rows = %d\n", d->sub_n_rows);
	cdw_vdm ("INFO: txt n_rows = %d\n", d->txt_n_rows);

	return CDW_OK;
}
