/**
 * @brief Hardware abstraction layer for ACPI power management
 *
 * This module contains all functions for ACPI based machines that
 * are used by the high level logic modules to do their job.
 *
 * This module is needed for all new i386 laptops and the MacBook
 * family from Apple.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/module_acpi.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>

#include <glib.h>
#include <pbb.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "module_acpi.h"
#include "support.h"
#include "debug.h"

#define SECTION "MODULE ACPI"
#define ACPI_BUFFER_LEN	50

#if defined(DEBUG) && ACPI_PATHS
void
acpi_print_file (GString *file, gpointer data)
{
	printf ("file: %s\n", file->str);
}
#endif

/**
 * @brief module specific data
 */
struct moddata_acpi {
	int fd_evacpi;     /* filehandle to receive ACPI events */
	struct modflags_acpi flags;
	GList *BatteryList;
	int timeleft;
	char *ac;       /* ACPI path to AC adaptor status */
	char *lid;      /* ACPI path to lid button status */
	GString *event_buffer;
} modbase_acpi;

/**
 * @brief Initalizes the module acpi
 *
 * This function initializes the module acpi. The data structure will
 * be initialized and the hardware status is read.
 *
 * Immideately after this initialisation the module is able to and 
 * will answer upon requests.
 *
 * This function will also collect batteries available in the system.
 * This is only done once. See acpi_update_batteries() for more details
 * why.
 *
 * @see acpi_update_batteries()
 *
 * @return  error code, zero if the module is ready to operate
 *
 * @todo Add real check and add functions to trigger sleep if
 *       requested.
 */
int
acpi_init ()
{
	struct moddata_acpi *base = &modbase_acpi;
	int val, rc;
	
	/* is ACPI available on this machine? */
	if ((rc = check_devorfile (PATH_ACPI"/info", TYPE_FILE))) {
		print_msg (PBB_ERR, _("Can't access ACPI interface in /proc. Are all necessary modules loaded?\n"));
		return rc;
	}

	/* check if sleep is supported on this system */
	base->flags.sleepsupported = 0;

	base->ac = NULL;
	base->lid = NULL;
	base->event_buffer   = g_string_sized_new (256);
	base->flags.ac_power = acpi_read_ac_adapter();

	/* collecting of batteries is only done once. See
	 * acpi_update_batteries() for more details why.
	 */
	base->BatteryList = acpi_collect_batteries (base->BatteryList);
	base->BatteryList = acpi_update_batteries (base->BatteryList);
	base->timeleft    = acpi_calc_timeleft (base->BatteryList);

#if !(defined(DEBUG) && SIMU_ACPI)
	if ((base->fd_evacpi = acpi_open_socket (ACPID_SOCKET)) < 0)
		if ((base->fd_evacpi = open(PATH_ACPI"/event", O_RDONLY)) < 0) {
			print_msg (PBB_ERR, _("Can't connect to ACPI event queue: %s\n"), strerror(errno));
			return E_OPEN;
		}
#else
	base->fd_evacpi = STDIN_FILENO;
#endif

	if ((addInputSource (base->fd_evacpi, acpi_event_handler, NULL, FALSE)) == NULL) {
		print_msg (PBB_ERR, _("Can't install ACPI event handler.\n"));
		return E_NOMEM;
	}

	register_function (T1000QUEUE, acpi_timer1000);
	register_function (QUERYQUEUE, acpi_query);
	register_function (CONFIGQUEUE, acpi_configure);
	return 0;
}

/**
 * @brief  Frees all ressources allocated from module acpi
 *
 * This function is usually called on program termination. It
 * frees all allocated ressources and closes all open file
 * handles.
 */
void
acpi_exit ()
{
	struct moddata_acpi *base = &modbase_acpi;
	GList *iter, *niter;
	
	iter =  g_list_first (base->BatteryList);
	while (iter) {
		niter = g_list_next (iter);
		acpi_free_battery ((struct battery *) iter->data);
		iter = niter;
	}

	if (base->ac)
		g_free (base->ac);
	
	if (base->lid)
		g_free (base->lid);

	if (base->fd_evacpi != -1)
		close (base->fd_evacpi); /* close ACPI event device */
}

/**
 * @brief  Request information from the module
 *
 * @see acpi_handle_tags()
 *
 * @param *taglist  Taglist containing one or more requests
 */
void
acpi_query (struct tagitem *taglist)
{
	acpi_handle_tags (MODE_QUERY, taglist);
}

/**
 * @brief  Configure parameters of the module
 *
 * @see acpi_handle_tags()
 *
 * @param *taglist  Taglist containing one or more parameters
 */
void
acpi_configure (struct tagitem *taglist)
{
	acpi_handle_tags (MODE_CONFIG, taglist);
}

/**
 * @brief  Get or set module parameters
 *
 * This function handles the module-2-module communication.
 * If a parameter should be read or written this function
 * is called. Return vales are written to the taglist directly.
 *
 * @param  cfgure  MODE_CONFIG if parameters should be set or
 *                 MODE_QUERY if parameters should be read.
 * @param  *taglist  Taglist with one ore more parameters.
 */
void
acpi_handle_tags (int cfgure, struct tagitem *taglist)
{
	struct moddata_acpi *base = &modbase_acpi;

	while (taglist->tag != TAG_END) {
		switch (taglist->tag) {
		case TAG_SLEEPSUPPORTED:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else		taglist->data = base->flags.sleepsupported;
			break;
#ifndef WITH_IBAM
		case TAG_TIMEREMAINING:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else		taglist->data = base->timeleft;
			break;
#endif
		case TAG_POWERSOURCE:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else		taglist->data = base->flags.ac_power;
			break;
		}
		taglist++;
	}
}

/**
 * @brief ACPI event handler (original design)
 *
 * @param  fd         Filehandle of ACPI event device
 * @param  user_data  Not used yet
 * @return TRUE, if the handler should stay active
 */
gboolean
acpi_event_handler (int fd, gpointer user_data)
{
	struct moddata_acpi *base = &modbase_acpi;
	struct tagitem taglist[2];
	struct acpi_event acpi;
	GString *buffer;
	gboolean status;
	char *cur, *next, *tmp;
	int n;

	taglist_init (taglist);
	buffer = g_string_sized_new (ACPI_BUFFER_LEN + 1);

	while ((n = read (fd, buffer->str, ACPI_BUFFER_LEN)) > 0) {
		g_string_append_len (base->event_buffer, buffer->str, n);
		if (n < ACPI_BUFFER_LEN)
			break;
	}

	if (n < 0) {
		/* An error occoured. If errno == EAGAIN, the received data
		 * has been thrown away by the socket driver and there is
		 * nothing left to read for us. We reset and try again next
		 * time.
		 */
		if (errno != EAGAIN)
			print_msg (PBB_ERR, _("Can't read ACPI events: %s.\n"), strerror(errno));
		goto out;
	}

	if ((tmp = index(base->event_buffer->str, '\n'))) {
		*tmp = 0;  /* terminate substring */
		buffer = g_string_assign (buffer, base->event_buffer->str);
		g_string_erase (base->event_buffer, 0, buffer->len + 1);

#if defined(DEBUG) && ACPI_EVENTS
		printf ("ACPI event received: '%s'\n", buffer->str);
#endif
		/* decode ACPI event */
		cur = buffer->str;
		for (n=0; n < 4; n++) {
			if ((next = strchr (cur, ' ')))
				*next++ = '\0';
			else if (n != 3)
				break;  /* syntax error */

			switch (n) {
				case 0:	acpi.device = cur; break;
				case 1: acpi.bus    = cur; break;
				case 2: acpi.type   = axtoi(cur); break;
				case 3: acpi.data   = axtoi(cur); break;
			}

			cur = next;
		}

		if (n == 4) {
#if defined(DEBUG) && ACPI_EVENTS
			printf ("  device: '%s'\n  bus   : '%s'\n  type  : %ld\n  data  : %ld\n\n",
				acpi.device, acpi.bus, acpi.type, acpi.data);
#endif
			if (strcmp ("button/lid", acpi.device) == 0) {
				/* The ACPI event doesn't tell us the status of the cover.
				 * Instead a count of "how often the lid button was pressed"
				 * is transmitted which is of no known use. We have to
				 * retrieve the cover status on our own here.
				 */
				status = acpi_read_cover_status ();
				taglist_add (taglist, TAG_COVERSTATUS, status ? 1 : 0);
			} else if (strcmp ("ac_adapter", acpi.device) == 0) {
				/* data is 0x00 if running on battery, 0x01 if running on
				 * AC and 0xff if power source is unknown. An unknown
				 * powersource is treated as AC here.
				 */
				taglist_add (taglist, TAG_POWERCHANGED, acpi.data);
				base->flags.ac_power = acpi.data;
			}

			/* distribute events to other modules */
			if (taglist[0].tag != TAG_END)
				process_queue (CONFIGQUEUE, taglist);
		}

#if defined(DEBUG) && ACPI_PATHS
		printf ("  base->ac  = %s\n", base->ac ? base->ac : "<unset>");
		printf ("  base->lid = %s\n", base->lid ? base->lid : "<unset>");
#endif
	}

out:
	g_string_free (buffer, TRUE);
	return TRUE;
}

/**
 * @brief  Time function called every second
 *
 * This function does regulary tasks like reporting
 * remaining battery time or flag changes.
 *
 * @param  taglist  not used in this function
 */
void
acpi_timer1000 (struct tagitem *taglist)
{
	struct moddata_acpi *base = &modbase_acpi;
	struct tagitem args[6];
	int val;

	taglist_init (args);

	val               = base->timeleft;
	base->BatteryList = acpi_update_batteries (base->BatteryList);
	base->timeleft    = acpi_calc_timeleft (base->BatteryList);

#if defined(DEBUG) && ACPI_BATTERY
	static int count = 0;
	int n = g_list_length (base->BatteryList);
	if (count != n) {
		acpi_print_battery (base->BatteryList);
		printf ("Time left: %d\n", base->timeleft);
		count = n;
	}
#endif

#ifndef WITH_IBAM
	if (val != base->timeleft && !base->flags.ac_power) {
		taglist_add (args, TAG_TIMECHANGED, base->timeleft);
	}
#endif

	if (args[0].tag != TAG_END)
		process_queue (CONFIGQUEUE, args);  /* distribute changes to other modules */
}

/**
 * @brief Calculate the time left on battery
 *
 * The battery list contains data about current and voltage
 * and this seperatly for each battery but the time the system
 * will run on battery was not stored. This function adds
 * the capacity of ll batteries in system and divide them by
 * the overall current to get a good guess how log the battery
 * will last.
 *
 * @param  batlist  BatteryList to calulate over
 * @return time remaining in seconds or -1 if not available
 */
int
acpi_calc_timeleft (GList *batlist)
{
	GList *iter;
	int charge, current;
	struct battery *bat;

	charge = current = 0;
	iter = g_list_first (batlist);
	while (iter) {
		bat = (struct battery *) iter->data;
		charge  += bat->capacity;
		current += bat->current;
		iter = g_list_next (iter);
	}
	return current == 0 ? -1 : charge * 3600 / current;
}

/**
 * @brief Update battery data in the battery list
 *
 * This function reads the battery status from /proc/acpi/battery
 * and fill the battery data fields in dthe battery list.
 *
 * Before updating the battery data, the battery list is rescanned
 * for batteries new in the system. This will be done by the
 * function acpi_collect_batteries().
 *
 * If a battery in the batery list couldn't be accessed, it will
 * be removed from the battery list.
 *
 * @see acpi_collect_batteries()
 *
 * @param  *batlist  List of batteries to update
 * @return revised and updated battery list
 */
GList *
acpi_update_batteries (GList *batlist)
{
	FILE *fd;
	char buffer[100], *token, *tmp;
	GList *iter, *niter;
	GString *batinfo;
	struct battery *bat;

	/* update battery list */
	/* having this call here will allow batteries to appear and
	 * vanish dynamically. The battery-list will catch up with
	 * the changes and is always up-to-date. I think this is
	 * not necessary because the ACPI system knows about all
	 * possible batteries from the beginning and won't add or
	 * remove battery entries on the fly. Therefor this call
	 * is commented out here and added to acpi_init()
	 *
	 * batlist = acpi_collect_batteries (batlist);
	 */

	iter = g_list_first(batlist);
	while (iter) {
		niter = g_list_next (iter);
		bat = (struct battery *) iter->data;

		if ((fd = fopen (bat->path, "r"))) {
			while (fgets (buffer,sizeof (buffer), fd)) {
				cleanup_buffer (buffer);
				if ((token = strtok (buffer, ":\n"))) {
					if (!strncmp ("chargingstate", token, 13)) {
						bat->charging = strcmp (strtok(0, "\n"), "charging") ? FALSE : TRUE;
					} else if (!strncmp ("presentrate", token, 11)) {
						bat->current = atoi (strtok(0,":\n"));
					} else if (!strncmp ("remainingcapacity", token, 17)) {
						bat->capacity = atoi (strtok(0, ":\n"));
					} else if (!strncmp ("presentvoltage", token, 14)) {
						bat->voltage = atoi (strtok(0,":\n"));
					} else if (!strncmp ("present", token, 7)) {
						bat->present = strcmp (strtok(0, "\n"), "yes") ? FALSE : TRUE;
					} else
						strtok (0,":\n");
				}
			}
			fclose(fd);

			batinfo = g_string_new (bat->path);
			if ((tmp = g_strrstr (batinfo->str,"/")) != NULL) {
				batinfo = g_string_truncate (batinfo, tmp - batinfo->str);
				batinfo = g_string_append (batinfo, "/info");
				if ((fd = fopen (batinfo->str, "r"))) {
					while (fgets (buffer,sizeof (buffer), fd)) {
						cleanup_buffer (buffer);
						if ((token = strtok (buffer, ":\n"))) {
							if (!strncmp ("lastfullcapacity", token, 16)) {
								bat->full_capacity = atoi (strtok(0,":\n"));
							} else
								strtok (0,":\n");
						}
					}
				}
				fclose(fd);
			}
			g_string_free (batinfo, TRUE);

		} else {
			batlist = g_list_remove (batlist, bat);
			acpi_free_battery (bat);
		}
	
		iter = niter;
	}

	return batlist;
}

/**
 * @brief  Frees all ressources of a battery structure
 *
 * This functions frees all allocated ressources of a battery
 * structure.
 *
 * @param *bat  Battery structure to clean up
 */
void
acpi_free_battery (struct battery *bat)
{
	if (bat->path)
		g_free(bat->path);
	g_free (bat);
}

/**
 * @brief finds out if the AC connector is plugged in
 *
 * This function checks if the AC connector is plugged in.
 *
 * If the status can't be read, the function returns TRUE, assuming
 * that running errornous on AC has less side effects that running
 * errornous on battery. A warning in the log file pointed that out.
 *
 * If more than one AC adapter is available, only the first one is
 * taken into account. Any further adapter will be ignored. A warning
 * in the log file will point this case out.
 *
 * @return  TRUE, if the machine is running on AC, false otherwise
 */
gboolean
acpi_read_ac_adapter ()
{
	struct moddata_acpi *base = &modbase_acpi;
	GList *files;
	GString *tmp;
	FILE *fd;
	char buffer[100], *token;
	gboolean rc = TRUE;

	if (base->ac == NULL) {
		if ((files = acpi_get_file_list ("ac_adapter"))) {
			if (g_list_length (files) > 1)	
				print_msg (PBB_WARN, _("More than one AC adapter found. Only the first one is used.\n"));
			tmp = (GString *) g_list_nth_data (files, 0);
			/* save first file path for later use */
			base->ac = g_strdup (tmp->str);
#if defined(DEBUG) && ACPI_PATHS
			g_list_foreach (files, (GFunc) acpi_print_file, NULL);
#endif
			acpi_free_file_list (files);
		}
	}

	if (base->ac) {
		if ((fd = fopen (base->ac,"r"))) {
			while (fgets (buffer, sizeof (buffer), fd)) {
				cleanup_buffer (buffer);
				if ((token = strtok (buffer,":\n"))) {
					if (!strncmp ("state", token, 5)) {
						rc = strcmp ("on-line", strtok(0,":\n")) == 0 ? TRUE : FALSE;
						break;
					} else if (!strncmp ("status", token, 6)) {
						rc = strcmp ("on-line", strtok(0,":\n")) == 0 ? TRUE : FALSE;
						break;
					} else
						strtok (0,":\n");
				}
			}
			fclose(fd);
		} else
			print_msg (PBB_ERR, _("Can't read AC adapter status - assuming it is connected.\n"));
	}
	
	return rc;
}

/**
 * @brief Reads the status of the lid switch
 *
 * This function checks if the cover is open or close
 *
 * If the status can't be read, the function returns TRUE, assuming
 * that the cover is open. Otherwise it wouldn't be possible to work
 * with the machine anymore if the wrong configuration had been set.
 *
 * @return  TRUE, if the cover is open, false otherwise
 */
gboolean
acpi_read_cover_status ()
{
	struct moddata_acpi *base = &modbase_acpi;
	GList *files;
	GString *tmp;
	FILE *fd;
	char buffer[100], *token;
	gboolean rc = TRUE;

	if (base->lid == NULL) {
		if ((files = acpi_get_file_list ("button/lid"))) {
			if (g_list_length (files) > 1)	
				print_msg (PBB_WARN, _("More than one Lid button found. Only the first one is used.\n"));
			tmp = (GString *) g_list_nth_data (files, 0);
			/* save first file path for later use */
			base->lid = g_strdup (tmp->str);
#if defined(DEBUG) && ACPI_PATHS
			g_list_foreach (files, (GFunc) acpi_print_file, NULL);
#endif
			acpi_free_file_list (files);
		}
	}

	if (base->lid) {
		if ((fd = fopen (base->lid,"r"))) {
			while (fgets (buffer, sizeof (buffer), fd)) {
				cleanup_buffer (buffer);
				if ((token = strtok (buffer,":\n"))) {
					if (!strncmp ("state", token, 5)) {
						rc = strncmp ("open", strtok(0,":\n"), 4) == 0 ? TRUE : FALSE;
						break;
					} else if (!strncmp ("status", token, 6)) {
						rc = strncmp ("open", strtok(0,":\n"), 4) == 0 ? TRUE : FALSE;
						break;
					} else
						strtok (0,":\n");
				}
			}
			fclose(fd);
		} else
			print_msg (PBB_ERR, _("Can't read cover status - assuming it is open.\n"));
	}

	return rc;
}


/**
 * @brief callback function to find a battery with a given path
 *
 * This callback function will be called for every battery in the battery
 * list and checks if this battery could be found on the given path. In
 * that case it returns zero.
 *
 * @param  *a   Battery entry to check
 * @param  *b   Battery path to find
 * @return zero, if the battery entry was found
 */
gint
acpi_find_battery (gconstpointer *a, gconstpointer *b)
{
	struct battery *bat     = (struct battery *) a;
	char           *batpath = (char *) b;

	return strcmp (batpath, bat->path);
}

/**
 * @brief Collects all batteries found in /proc/acpi/battery
 *
 * This function collects all batteries from /proc/acpi/battery
 * and creates a data field for each of them. This datafields
 * will be queued up in a list.
 *
 * This function fills only the path entry of the datafield. Any
 * other field will be filled by acpi_update_batteries().
 *
 * If an old battery list was given, the function adds newly found
 * batteries to this list and ignore batteries already in the list.
 *
 * @see acpi_update_batteries()
 *
 * @param *batlist  Existing batlist or NULL if a new should be
 *                  generated
 * @return new or revised battery list
 */
GList *
acpi_collect_batteries (GList *batlist)
{
	DIR *dd;
	struct dirent *dir;
	GString *batpath;
	struct battery *newbat;

	batpath = g_string_new (NULL);

	if ((dd = opendir (PATH_ACPI"/battery"))) {
		while ((dir = readdir (dd))) {
			if (dir->d_name[0] == '.')
				continue;  /* skip "." and ".." */
			
			g_string_printf (batpath, PATH_ACPI"/battery/%s/state", dir->d_name);
			if (access(batpath->str, R_OK)) {
				g_string_printf (batpath, PATH_ACPI"/battery/%s/status", dir->d_name);
				if (access(batpath->str, R_OK))
					continue;  /* try next one */
			}

			if ((g_list_find_custom (batlist, batpath->str, (GCompareFunc) acpi_find_battery)) == NULL) {
				newbat = g_new0 (struct battery, 1);
				newbat->path = g_strdup (batpath->str);
				batlist = g_list_append (batlist, newbat);
			}
		}
		closedir (dd);
	}
	return batlist;
}

/**
 * @brief Opens the acpid socked for reading events
 *
 * The ACPI system reports events through /proc/acpi/event but
 * it allows only one listener. If another daemon already listens
 * for events it is impossible for pbbuttonsd to receive ACPI
 * events.
 *
 * The ACPI deamon acpid work around this problem by forwarding
 * the events to a newly created socket file. This way multiply
 * daemons could cascade on /proc/acpi/event.
 *
 * This function opens the socket file from acpid and prepare
 * it so that pbbuttonsd can read events from it.
 *
 * @param   *name  Name of the socket file
 * @return  filehandle of open socket or -1 on error
 */
int
acpi_open_socket (char *name)
{
	struct sockaddr_un addr;
	int fd, val;

	if ((fd = socket(AF_UNIX, SOCK_STREAM, 0)) >= 0) {
		bzero (&addr, sizeof(addr));
		addr.sun_family = AF_UNIX;
		sprintf(addr.sun_path, "%s", name);

		if ((connect(fd, (struct sockaddr *)&addr, sizeof(addr))) >= 0) {
			/* Make socket non-blocking because select() may spuriously
			 * report sockets as ready. See man page of select() for
			 * details.
			 */
			val = fcntl(fd, F_GETFL);
			fcntl(fd, F_SETFL, val | O_NONBLOCK);
			return fd;
		}
	
		close (fd);
	}

	return -1;
}

/**
 * @brief  scans a directory and returns all 'state' and 'status' files found
 *
 * This function scans a given directory for the files 'state' and 'status'
 * and returns a list with all found and valid path names. A path name is valid
 * if the file exists and the caller has the right permissions to read them.
 *
 * The list must be freed with acpi_free_file_list() after usage.
 *
 * @see acpi_free_file_list()
 *
 * @param  *startdir  subdirectory in /proc/acpi to scan
 * @return  GList with all found 'state' and 'status' files or NULL if the
 *          directory does not exist or no files could be found.
 */
GList*
acpi_get_file_list (char *startdir)
{
	GString *path, *file;
	GList *list;
	DIR *dd;
	struct dirent *dir;

	list = NULL;
	path = g_string_new (PATH_ACPI"/");
	path = g_string_append (path, startdir);

	if ((dd = opendir (path->str))) {
		while ((dir = readdir (dd))) {
			if (dir->d_name[0] == '.')
				continue;  /* skip "." and ".." */
			
			file = g_string_new (NULL);
			g_string_printf (file, "%s/%s/state", path->str, dir->d_name);
			if (access(file->str, R_OK)) {
				g_string_printf (file, "%s/%s/status", path->str, dir->d_name);
				if (access(file->str, R_OK)) {
					g_string_free (file, TRUE);
					continue;  /* try next one */
				}
			}

			list = g_list_append (list, file);
		}
		closedir (dd);
	}
	return list;
}

/**
 * @brief  handler for acpi_free_file_list()
 *
 * This function will be called by acpi_free_file_list() for
 * every list entry. The GString of the list entry will be
 * freed.
 *
 * @see acpi_free_file_list()
 *
 * @param  *file   GString that should be freed
 * @param  data    Not used in this function
 */
void
acpi_free_file (GString *file, gpointer data)
{
	g_string_free (file, TRUE);
}

/**
 * @brief  frees all ressources of a file list
 *
 * This function frees all ressources of a file list created
 * with acpi_get_file_list(). The function acpi_free_file()
 * is called for every list entry.
 *
 * @see acpi_get_file_list(), acpi_free_file()
 *
 * @param  *list   Pointer to GList that should be freed
 */
void
acpi_free_file_list (GList *list)
{
	g_list_foreach (list, (GFunc) acpi_free_file, NULL);
	g_list_free (list);
}

#if defined(DEBUG) && ACPI_BATTERY
void
acpi_print_battery (GList *batlist)
{
	struct battery *bat;
	GList *iter;

	printf ("\nBattery List\n------------\n");
	iter = g_list_first(batlist);
	while (iter) {
		bat = (struct battery *) iter->data;
		
		printf ("Battery: %s\n", bat->path);
		printf ("  (%p)\n", bat);
		printf ("  present:       %s\n", bat->present ? "yes" : "no");
		printf ("  charging:      %s\n", bat->charging ? "yes" : "no");
		printf ("  full capacity: %d mAh\n", bat->full_capacity);
		printf ("  capacity:      %d mAh\n", bat->capacity);
		printf ("  voltage:       %d mV\n", bat->voltage);
		printf ("  current:       %d mA\n\n", bat->current);

		iter = g_list_next (iter);
	}
}
#endif

